<?php
session_start();
require_once 'config.php'; // Ensure this has $conn and helper functions
require_once 'includes/emission_factors_config.php'; // Or 'includes/emission_factors_config.php'

// --- Essential Helper Functions (ensure these are in config.php or defined here) ---
if (!function_exists('sanitize_input')) {
    function sanitize_input($data) {
        if (is_array($data)) {
            return array_map('sanitize_input', $data);
        }
        $data = trim($data);
        $data = stripslashes($data);
        $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
        return $data;
    }
}
if (!function_exists('is_post_request')) {
    function is_post_request() {
        return $_SERVER['REQUEST_METHOD'] === 'POST';
    }
}
if (!function_exists('redirect_to')) {
    function redirect_to($location) {
        header("Location: " . $location);
        exit;
    }
}
if (!function_exists('set_session_message')) {
    function set_session_message($message, $type = 'info') {
        $_SESSION['flash_message'] = ['message' => $message, 'type' => $type];
    }
}
if (!function_exists('display_session_message')) {
    function display_session_message() {
        if (isset($_SESSION['flash_message'])) {
            $messageData = $_SESSION['flash_message'];
            echo '<div class="alert alert-' . htmlspecialchars($messageData['type']) . '" style="padding: 10px; margin-bottom: 15px; border: 1px solid #ccc;">' . htmlspecialchars($messageData['message']) . '</div>';
            unset($_SESSION['flash_message']);
        }
    }
}
// --- End Helper Functions ---


// Check if general information has been submitted
if (!isset($_SESSION['general_id'])) {
    set_session_message("Please fill out the general information first.", "warning");
    redirect_to("general.php"); // Or your general info page
}
// Optionally, check if Scope 1 was completed if it's a prerequisite
// if (!isset($_SESSION['scope1_total'])) {
//    set_session_message("Please complete Scope 1 emissions first.", "warning");
//    redirect_to("scope1.php");
// }

$general_id = $_SESSION['general_id'];
$errors = [];
$success = false;

// Initialize variables for form data and calculated totals
$electricity_regions_post = $_POST['electricity_region'] ?? [];
$electricity_units_post = $_POST['electricity_unit'] ?? []; // For electricity unit
$electricity_qty_2021_post = $_POST['electricity_qty_2021'] ?? [];
$electricity_qty_2022_post = $_POST['electricity_qty_2022'] ?? [];
$electricity_qty_2023_post = $_POST['electricity_qty_2023'] ?? [];
$electricity_qty_2024_post = $_POST['electricity_qty_2024'] ?? [];
$electricity_total_co2e = 0;

$include_shc_post = isset($_POST['include_shc']) && $_POST['include_shc'] === '1';
$shc_types_post = $_POST['shc_type'] ?? [];
$shc_qty_2021_post = $_POST['shc_qty_2021'] ?? [];
$shc_qty_2022_post = $_POST['shc_qty_2022'] ?? [];
$shc_qty_2023_post = $_POST['shc_qty_2023'] ?? [];
$shc_qty_2024_post = $_POST['shc_qty_2024'] ?? [];

$years = [2021, 2022, 2023, 2024]; // Define $years array
$year_column_keys = ['col1', 'col2', 'col3', 'col4'];

if (is_post_request()) {
    mysqli_begin_transaction($conn);
    try {
        $electricity_total_co2e = 0;
        $shc_total_co2e = 0;

        // --- Purchased Electricity Calculation ---		
        for ($i = 0; $i < count($electricity_regions_post); $i++) {
            $region = sanitize_input($electricity_regions_post[$i]);
            $unit = sanitize_input($electricity_units_post[$i]); 
            if (empty($region) || empty($unit)) {
                continue;
            }

            $co2e_for_this_row = 0;
            foreach (array_keys($years) as $year_index) { // Iterate 0, 1, 2, 3
                $current_year_value = $years[$year_index]; // Actual year: 2021, 2022...
                $ef_key_for_config = $year_column_keys[$year_index]; // 'col1', 'col2'...

                // Dynamically get the quantity for the current year from the correct $_POST array
                $quantity_for_year = 0;
                switch ($current_year_value) {
                    case 2021: $quantity_for_year = floatval($electricity_qty_2021_post[$i] ?? 0); break;
                    case 2022: $quantity_for_year = floatval($electricity_qty_2022_post[$i] ?? 0); break;
                    case 2023: $quantity_for_year = floatval($electricity_qty_2023_post[$i] ?? 0); break;
                    case 2024: $quantity_for_year = floatval($electricity_qty_2024_post[$i] ?? 0); break;
                }

                if ($quantity_for_year > 0) {
                    $ef = null; 
                    if (isset($emission_factors_data_scope2['electricity'][$region]) &&
                        isset($emission_factors_data_scope2['electricity'][$region][$unit]) &&
                        isset($emission_factors_data_scope2['electricity'][$region][$unit][$ef_key_for_config])) { // Use $ef_key_for_config
                        $ef = $emission_factors_data_scope2['electricity'][$region][$unit][$ef_key_for_config];
                    }
                    
                    if ($ef === null) {
                        $errors[] = "Electricity: EF not found for region '$region', unit '$unit', column_key '$ef_key_for_config' (for year '$current_year_value').";
                        // error_log("Electricity: EF not found for region '$region', unit '$unit', column_key '$ef_key_for_config' (for year '$current_year_value').");
                        continue; // Skip this year's calculation for this row if EF is missing
                    }
                    $co2e_for_this_row += ($quantity_for_year * $ef); // Assuming EF is tCO2e/unit
                }
            }
            $electricity_total_co2e += $co2e_for_this_row;
        }

        // --- Purchased Steam, Heat & Cooling Calculation ---
        if ($include_shc_post) {
            for ($i = 0; $i < count($shc_types_post); $i++) {
                $type = sanitize_input($shc_types_post[$i]);
                $unit_shc = 'metric tonnes'; 
                if (empty($type)) continue;

                // V V V V V  DEFINE $quantities_for_row_shc INSIDE THIS LOOP V V V V V
                $quantities_for_row_shc = [
                    2021 => floatval($shc_qty_2021_post[$i] ?? 0),
                    2022 => floatval($shc_qty_2022_post[$i] ?? 0),
                    2023 => floatval($shc_qty_2023_post[$i] ?? 0),
                    2024 => floatval($shc_qty_2024_post[$i] ?? 0),
                ];
                // ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^

                $ef = $emission_factors_data_scope2['steam_heat_cooling'][$type][$unit_shc] ?? null;
                if ($ef === null) { $errors[] = "S/H/C: EF not found for type '$type'."; continue; }
                
                $co2e_for_this_shc_row = 0;
                foreach ($years as $year) { // Use the defined $years array
                    $quantity_for_year = $quantities_for_row_shc[$year]; // Use specific variable
                    if ($quantity_for_year > 0) {
                        // Assuming S/H/C EFs might not be year-specific, 
                        // but if they are, add [$year] to the EF lookup here too.
                        // For now, assuming EF is not year specific for S/H/C as per config.
                        $co2e_for_this_shc_row += ($quantity_for_year * $ef); // Assuming EF is tCO2e/unit
                    }
                }
                $shc_total_co2e += $co2e_for_this_shc_row;
            }
        }
        
        if (!empty($errors)) {
            // Important: If errors occurred, we might not want to proceed with DB commit.
            // Throwing an exception will trigger the catch block and rollback.
            throw new Exception("Input data validation/EF lookup errors occurred: " . implode("; ", $errors));
        }

        $total_scope2_emissions = $electricity_total_co2e + $shc_total_co2e;

        // --- Database: Insert into scope2_emissions (Main Table) ---
        // This part should be correct as it uses the calculated totals
        $sql_main = "INSERT INTO scope2_emissions (general_id, electricity_total_co2e, steam_heat_cooling_total_co2e, include_steam_heat_cooling, total_scope2_emissions) VALUES (?, ?, ?, ?, ?)";
        if ($stmt_main = mysqli_prepare($conn, $sql_main)) {
            $include_shc_db_val = $include_shc_post ? 1 : 0;
            mysqli_stmt_bind_param($stmt_main, "iddid", $general_id, $electricity_total_co2e, $shc_total_co2e, $include_shc_db_val, $total_scope2_emissions);
			
            if (!mysqli_stmt_execute($stmt_main)) {
                throw new Exception("Error executing main Scope 2 statement: " . mysqli_stmt_error($stmt_main));
            }
            $scope2_emission_id = mysqli_insert_id($conn);
            mysqli_stmt_close($stmt_main);
        } else {
            throw new Exception("Error preparing main Scope 2 statement: " . mysqli_error($conn));
        }

        // --- Database: Insert Electricity Consumption Details ---
        $sql_elec_detail = "INSERT INTO electricity_consumption_details (scope2_emission_id, region, input_unit, quantity_2021, quantity_2022, quantity_2023, quantity_2024) VALUES (?, ?, ?, ?, ?, ?, ?)";
        if ($stmt_elec_detail = mysqli_prepare($conn, $sql_elec_detail)) {
            for ($i = 0; $i < count($electricity_regions_post); $i++) {
                $region_e = sanitize_input($electricity_regions_post[$i]);
                $unit_e = isset($electricity_units_post[$i]) ? sanitize_input($electricity_units_post[$i]) : '';
                if (empty($region_e) || empty($unit_e)) continue;

                $q21_e = floatval($electricity_qty_2021_post[$i]??0); 
                $q22_e = floatval($electricity_qty_2022_post[$i]??0);
                $q23_e = floatval($electricity_qty_2023_post[$i]??0);
                $q24_e = floatval($electricity_qty_2024_post[$i]??0);
                
                if (($q21_e + $q22_e + $q23_e + $q24_e) > 0) {
                    // Correct type string: 1 int, 2 strings, 4 doubles
                    mysqli_stmt_bind_param($stmt_elec_detail, "issdddd", 
                        $scope2_emission_id, $region_e, $unit_e,
                        $q21_e, $q22_e, $q23_e, $q24_e
                    );
                    if (!mysqli_stmt_execute($stmt_elec_detail)) throw new Exception("DB Error (elec detail): " . mysqli_stmt_error($stmt_elec_detail));
                }
            }
            mysqli_stmt_close($stmt_elec_detail);
        } else throw new Exception("DB Prepare Error (elec detail): " . mysqli_error($conn));
        
        // --- Database: Insert Steam, Heat & Cooling Details (only if included) ---
        if ($include_shc_post) {
            $sql_shc_detail = "INSERT INTO steam_heat_cooling_details (scope2_emission_id, energy_type, quantity_2021, quantity_2022, quantity_2023, quantity_2024) VALUES (?, ?, ?, ?, ?, ?)";
            if ($stmt_shc_detail = mysqli_prepare($conn, $sql_shc_detail)) {
                for ($i = 0; $i < count($shc_types_post); $i++) {
                    $type_shc = sanitize_input($shc_types_post[$i]);
                    if (empty($type_shc)) continue;

                    $q21_shc = floatval($shc_qty_2021_post[$i] ?? 0); $q22_shc = floatval($shc_qty_2022_post[$i] ?? 0);
                    $q23_shc = floatval($shc_qty_2023_post[$i] ?? 0); $q24_shc = floatval($shc_qty_2024_post[$i] ?? 0);

                    if (($q21_shc + $q22_shc + $q23_shc + $q24_shc) > 0) {
                        mysqli_stmt_bind_param($stmt_shc_detail, "isdddd",
                            $scope2_emission_id, $type_shc,
                            $q21_shc, $q22_shc, $q23_shc, $q24_shc
                        );
                        if (!mysqli_stmt_execute($stmt_shc_detail)) {
                            throw new Exception("Error inserting steam/heat/cooling detail for " . htmlspecialchars($type_shc) . ": " . mysqli_stmt_error($stmt_shc_detail));
                        }
                    }
                }
                mysqli_stmt_close($stmt_shc_detail);
            } else {
                throw new Exception("Error preparing steam/heat/cooling detail statement: " . mysqli_error($conn));
            }
        }

        mysqli_commit($conn);
        $success = true;
        $_SESSION['scope2_total'] = $total_scope2_emissions; // For potential use on next page/output
        //set_session_message("Scope 2 emissions saved successfully!", "success");
        redirect_to("intensity.php"); // Or your next page (e.g., Output page)

    } catch (Exception $e) {
        mysqli_rollback($conn);
        // Preserve existing errors and add the new exception message
        $current_errors = isset($errors) && is_array($errors) ? $errors : [];
        $exception_message = $e->getMessage();
        if(!in_array($exception_message, $current_errors)){ 
            $current_errors[] = $exception_message;
        }
        $errors = $current_errors;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>GHG Calculator - Scope 2</title>
	<link rel="stylesheet" href="assets/css/main.css" />
	<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
    <style>
        .alert { padding: 15px; margin-bottom: 20px; border: 1px solid transparent; border-radius: 4px; }
        .alert-danger, .errors { color: #a94442; background-color: #f2dede; border-color: #ebccd1; }
        .alert-success { color: #3c763d; background-color: #dff0d8; border-color: #d6e9c6; }
        .alert-warning { color: #8a6d3b; background-color: #fcf8e3; border-color: #faebcc; }
        .errors ul { margin-top: 0; margin-bottom: 0; padding-left: 20px;}
    </style>
    <script>
        // Pass PHP data to JavaScript (though not strictly needed for unit display in Scope 2 as it's static per selection)
        // const emissionFactorsConfigScope2 = <?php //echo json_encode($emission_factors_data_scope2); ?>;
    </script>
</head>
<body>
	<div id="page-wrapper">
		<header id="header">
			<div id="logo">
				<a href="index.html"><img src="images/cmm-logo.png" alt=""/></a>
			</div>
			<nav id="nav">
				<ul>
					<li><a href="index.html">Home</a></li>
					<li><a href="methodology.html">Methodology</a>
						<ul>
							<li><a href="#">User Guide</a></li>
						</ul>
					</li>
					<li><a href="about.html">About</a></li>
					<li><a href="general.php" class="button">Get Started</a></li>
				</ul>
			</nav>
		</header>
		<section id="form-page">
			<div class="wrapper">
				<div class="container">
					<div id="formbox" class="general">
						<div id="breadcrumb">
							<ul class="breadcrumb">
								<li><a>General</a></li>
								<li><a>Scope 1</a></li>
								<li class="active"><a>Scope 2</a></li>
								<li><a>Intensity</a></li>
								<li><a>Result Summary</a></li>
							</ul>
						</div>
                        <?php display_session_message(); ?>
						<?php if (!empty($errors)): ?>
							<div class="errors"><strong>Please correct the following errors:</strong><ul><?php foreach ($errors as $error): ?><li><?php echo htmlspecialchars($error); ?></li><?php endforeach; ?></ul></div>
						<?php endif; ?>
						
						<div class="note align-left">
							<div class="midtitle">Calculate Scope 2</div>
							<p>Scope 2 emissions are indirect GHG emissions resulting from the energy (e.g., electricity, steam, heat, and cooling) purchased by a company from third-party providers. In this calculator, we focus only on emissions from purchased electricity, as it is unlikely that SMEs in Malaysia purchase steam, heat, or cooling.</p>
							<p>Please provide as much annual data as possible for each electricity grid you purchase from, so the results can reflect how your emissions have changed over time. If data is unavailable for certain years, you may leave those fields blank.</p>
						</div>
						
						<form method="post" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" id="scope2-form">
							<div class="box">
                                <!-- Purchased Electricity Section -->
								<div class="smalltitle">Purchased Electricity
									<span class="info-container">
										<span class="info-icon">i</span>
										<div class="info-text">Amount of electricity purchased from grids in Malaysia in each reporting year. You may add up to three entries, one each for Peninsular, Sabah and/or Sarawak depending on how many regions you have operations in.</div>
									</span>
								</div>
								<div id="electricity-rows-container" class="scopeone">
                                    <?php
                                    $num_elec_rows = (is_post_request() && !empty($electricity_regions_post)) ? count($electricity_regions_post) : 1;
                                    for ($e_idx = 0; $e_idx < $num_elec_rows; $e_idx++):
                                        $current_elec_region = $electricity_regions_post[$e_idx] ?? '';
                                        $current_elec_unit = $electricity_units_post[$e_idx] ?? 'kWh';
                                        $current_elec_q21 = $electricity_qty_2021_post[$e_idx] ?? '';
                                        $current_elec_q22 = $electricity_qty_2022_post[$e_idx] ?? '';
                                        $current_elec_q23 = $electricity_qty_2023_post[$e_idx] ?? '';
                                        $current_elec_q24 = $electricity_qty_2024_post[$e_idx] ?? '';
                                    ?>
									<div class="row electricity-row">
										<div class="6u 12u(large)"> <!-- Adjust column classes as per your main.css grid -->
											<div class="selectoption" style="display: flex; align-items: center;">
												<label style="flex-grow: 1;"><strong>Region</strong>
													<select name="electricity_region[]">
														<option value="">-- Select Region --</option>
                                                        <?php foreach (array_keys($emission_factors_data_scope2['electricity']) as $region_option): ?>
                                                        <option value="<?php echo htmlspecialchars($region_option); ?>" <?php if ($current_elec_region === $region_option) echo 'selected'; ?>>
                                                            <?php echo htmlspecialchars($region_option); ?>
                                                        </option>
                                                        <?php endforeach; ?>
													</select>
												</label>
                                                <label><strong>Unit</strong> <!-- New Unit Select -->
                                                    <select name="electricity_unit[]" class="input-unit-select">
                                                        <option value="kWh" <?php if ($current_elec_unit === 'kWh') echo 'selected'; ?>>kWh</option>
                                                        <option value="MWh" <?php if ($current_elec_unit === 'MWh') echo 'selected'; ?>>MWh</option>
                                                    </select>
                                                </label>
											</div>
										</div>
										<div class="6u 12u(large)">
											<div class="input-year">	
												<div class="year"><label><strong>2021</strong></label><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="electricity_qty_2021[]" placeholder="kWh" value="<?php echo htmlspecialchars($current_elec_q21); ?>"></div>
												<div class="year"><label><strong>2022</strong></label><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="electricity_qty_2022[]" placeholder="kWh" value="<?php echo htmlspecialchars($current_elec_q22); ?>"></div>
												<div class="year"><label><strong>2023</strong></label><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="electricity_qty_2023[]" placeholder="kWh" value="<?php echo htmlspecialchars($current_elec_q23); ?>"></div>
												<div class="year"><label><strong>2024</strong></label><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="electricity_qty_2024[]" placeholder="kWh" value="<?php echo htmlspecialchars($current_elec_q24); ?>"></div>
											</div>
										</div>
                                        <?php if ($e_idx > 0) : ?>
										<div class="delete" style="width:100%; text-align:right; margin-top: -15px; margin-bottom: 10px;">
											<button type="button" class="remove-row"><img src="images/delete-icon-grey.png" alt="Remove"/></button>
										</div>
										<?php endif; ?>
									</div>
                                    <?php endfor; ?>
                                </div> <!-- end #shc_section_wrapper -->
								<div class="add"><button type="button" id="add-electricity-row">+ Add purchased electricity from another grid</button></div>
							</div> <!-- end .box -->
							<div id="action-btn">
								<a href="scope1.php"><button type="button" class="previous">Previous</button></a>
								<button type="submit" class="btn btn-primary next">Next</button>
							</div>
						</form>
					</div>
				</div>
			</div>
		</section>
		<footer id="footer">Copyright © 2025 Capital Markets Malaysia. All Rights Reserved.</footer>
	</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const MAX_ELECTRICITY_ROWS = 3; 
    const MAX_SHC_ROWS = 2;         

    function checkRowLimit(containerId, addButtonId, maxRows, specificRowClass) {
        const container = document.getElementById(containerId);
        const addButton = document.getElementById(addButtonId);
        if (!container || !addButton) return;
        const currentRowCount = container.querySelectorAll('.' + specificRowClass).length;
        if (currentRowCount >= maxRows) {
            addButton.disabled = true; addButton.style.opacity = 0.6; addButton.title = 'Maximum rows reached';
        } else {
            addButton.disabled = false; addButton.style.opacity = 1; addButton.title = '';
        }
    }

    function updatePlaceholdersForScope2Row(rowElement) {
        const unitSelect = rowElement.querySelector('.input-unit-select'); // For electricity
        const typeSelect = rowElement.querySelector('.scope2-type-select'); // For SHC (or electricity region if no separate unit select)
        const yearInputs = rowElement.querySelectorAll('.year-quantity-input');
        let placeholderUnit = "Qty"; 

        if (unitSelect && unitSelect.value) { // If there's an explicit unit select (like for electricity now)
            placeholderUnit = unitSelect.value;
        } else if (typeSelect && typeSelect.dataset.unit) { // Fallback to data-unit on type select (for SHC)
             placeholderUnit = typeSelect.dataset.unit;
        }
        
        yearInputs.forEach(input => { input.placeholder = placeholderUnit; });
    }
    
    const mainFormContainerForScope2 = document.getElementById('formbox') || document.getElementById('page-wrapper');
    if(mainFormContainerForScope2) {
        mainFormContainerForScope2.addEventListener('change', function(e) {
            // For electricity, when region OR unit changes, update placeholder
            if (e.target && (e.target.matches('select[name="electricity_region[]"]') || e.target.matches('select[name="electricity_unit[]"]'))) {
                const row = e.target.closest('.row.electricity-row');
                if (row) updatePlaceholdersForScope2Row(row);
            } 
            // For SHC, when type changes, update placeholder
            else if (e.target && e.target.matches('select[name="shc_type[]"]')) {
                const row = e.target.closest('.row.shc-row');
                if (row) updatePlaceholdersForScope2Row(row);
            }
        });
    }

    // Initial placeholder update for existing rows
    document.querySelectorAll('.electricity-row, .shc-row').forEach(row => {
        updatePlaceholdersForScope2Row(row);
    });

    // --- Row Templates for JS (Scope 2) ---
    const electricityRowTemplate = `
    <div class="row electricity-row">
        <div class="6u 12u(large)">
            <div class="selectoption">
                <label>
                    <select name="electricity_region[]" class="scope2-type-select" data-category="electricity"> <!-- data-category if using a generic updateUnitOptions -->
                        <option value="">-- Select Region --</option>
                        <?php if(isset($emission_factors_data_scope2['electricity'])) { foreach (array_keys($emission_factors_data_scope2['electricity']) as $region_option): ?>
                        <option value="<?php echo htmlspecialchars($region_option); ?>"><?php echo htmlspecialchars($region_option); ?></option>
                        <?php endforeach; } ?>
                    </select>
                </label>
                <label>
                    <select name="electricity_unit[]" class="input-unit-select">
                        <option value="kWh">kWh</option>
                        <option value="MWh">MWh</option>
                    </select>
                </label>
            </div>
        </div>
        <div class="6u 12u(large)">
            <div class="input-year">    
                <div class="year"><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="electricity_qty_2021[]" placeholder="kWh"></div>
                <div class="year"><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="electricity_qty_2022[]" placeholder="kWh"></div>
                <div class="year"><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="electricity_qty_2023[]" placeholder="kWh"></div>
                <div class="year"><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="electricity_qty_2024[]" placeholder="kWh"></div>
            </div>
        </div>
        <div class="delete" style="float:right; margin: -60px 80px 0px 0px"><button type="button" class="remove-row"><img src="images/delete-icon-grey.png" alt="Remove"/></button></div>
    </div>`;

    const shcRowTemplate = `
    <div class="row shc-row">
        <div class="6u 12u(large)">
            <div class="selectoption">
                <label style="flex-grow: 1;">
                    <select name="shc_type[]" class="scope2-type-select" data-category="steam_heat_cooling" data-unit="metric tonnes">
                        <option value="">-- Select Type --</option>
                        <?php if(isset($emission_factors_data_scope2['steam_heat_cooling'])) { foreach (array_keys($emission_factors_data_scope2['steam_heat_cooling']) as $shc_option): ?>
                        <option value="<?php echo htmlspecialchars($shc_option); ?>"><?php echo htmlspecialchars($shc_option); ?></option>
                        <?php endforeach; } ?>
                    </select>
                </label>
                <span class="unit-display energy" style="margin: 0px 0px 20px 0px">metric tonnes</span>
            </div>
        </div>
        <div class="6u 12u(large)">
            <div class="input-year">    
                <div class="year"><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="shc_qty_2021[]" placeholder="metric tonnes"></div>
                <div class="year"><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="shc_qty_2022[]" placeholder="metric tonnes"></div>
                <div class="year"><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="shc_qty_2023[]" placeholder="metric tonnes"></div>
                <div class="year"><input type="number" step="0.01" class="year-quantity-input year-input-scope2" name="shc_qty_2024[]" placeholder="metric tonnes"></div>
            </div>
        </div>
        <div class="delete" style="float:right; margin: -60px 80px 0px 0px"><button type="button" class="remove-row"><img src="images/delete-icon-grey.png" alt="Remove"/></button></div>
    </div>`;

    function setupAddRowButtonScope2(buttonId, containerId, rowTemplate, maxRows, specificRowClass) { /* ... same as before ... */ 
        const addButton = document.getElementById(buttonId);
        const container = document.getElementById(containerId);
        if (addButton && container) {
            addButton.addEventListener('click', function() {
                if (container.querySelectorAll('.' + specificRowClass).length < maxRows) {
                    container.insertAdjacentHTML('beforeend', rowTemplate);
                    const newRows = container.querySelectorAll('.' + specificRowClass);
                    const lastNewRow = newRows[newRows.length - 1];
                    if(lastNewRow) updatePlaceholdersForScope2Row(lastNewRow);
                    checkRowLimit(containerId, buttonId, maxRows, specificRowClass);
                } else { alert('Maximum ' + maxRows + ' entries reached.'); }
            });
            checkRowLimit(containerId, buttonId, maxRows, specificRowClass);
        }
    }
    setupAddRowButtonScope2('add-electricity-row', 'electricity-rows-container', electricityRowTemplate, MAX_ELECTRICITY_ROWS, 'electricity-row');
    setupAddRowButtonScope2('add-shc-row', 'shc-rows-container', shcRowTemplate, MAX_SHC_ROWS, 'shc-row');

    // Universal remove button handler
    if (mainFormContainerForScope2) { /* ... same as before, ensure checkRowLimit gets specificRowClass ... */ 
        mainFormContainerForScope2.addEventListener('click', function(e){
            const removeButton = e.target.closest('.remove-row');
            if(removeButton){ /* ... existing remove logic ... ensure it calls checkRowLimit correctly ... */
                const rowToRemove = removeButton.closest('.row');
                if (!rowToRemove) return;
                const container = rowToRemove.parentElement;
                if (!container) return;

                let containerId = container.id;
                let addButtonId = '', maxRows = 0, rowClassSelector = '';
                
                if (containerId === 'electricity-rows-container') {
                    addButtonId = 'add-electricity-row'; maxRows = MAX_ELECTRICITY_ROWS; rowClassSelector = 'electricity-row';
                } else if (containerId === 'shc-rows-container') {
                    addButtonId = 'add-shc-row'; maxRows = MAX_SHC_ROWS; rowClassSelector = 'shc-row';
                }

                if (container.querySelectorAll('.' + rowClassSelector).length > 1) {
                     rowToRemove.remove();
                } else {
                    rowToRemove.querySelectorAll('input[type="number"], select').forEach(input => input.value = '');
                    updatePlaceholdersForScope2Row(rowToRemove); // Reset placeholders for the cleared row
                }

                if (addButtonId) {
                    checkRowLimit(containerId, addButtonId, maxRows, rowClassSelector);
                }
            }
        });
    }

    const shcCheckbox = document.getElementById('toggle_shc_section');
    const shcSectionWrapper = document.getElementById('shc_section_wrapper');
    function toggleShcSection() { /* ... */  if (shcCheckbox && shcSectionWrapper) { shcSectionWrapper.style.display = shcCheckbox.checked ? '' : 'none'; } }
    if (shcCheckbox) { shcCheckbox.addEventListener('change', toggleShcSection); toggleShcSection(); }
});
</script>
<script src="assets/js/jquery.min.js"></script>
<script src="assets/js/jquery.scrolly.min.js"></script>
<script src="assets/js/jquery.dropotron.min.js"></script>
<script src="assets/js/jquery.scrollex.min.js"></script>
<script src="assets/js/skel.min.js"></script>
<script src="assets/js/util.js"></script>
<!--[if lte IE 8]><script src="assets/js/ie/respond.min.js"></script><![endif]-->
<script src="assets/js/main.js"></script>
</body>
</html>