<?php
session_start();
require_once 'config.php'; // Your DB connection and helper functions
require_once 'includes/emission_factors_config.php'; // This defines $emission_factors_data and $emission_factors_data_scope2

// --- Essential Helper Functions (ensure these are in config.php or defined here) ---
if (!function_exists('sanitize_input')) {
    function sanitize_input($data) {
		if (is_array($data)) { 
			return array_map('sanitize_input', $data); 
		} 
		$data = trim($data); 
		$data = stripslashes($data); 
		$data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8'); 
		return $data;
	}
}

if (!function_exists('display_session_message')) {
    function display_session_message() {
		if (isset($_SESSION['flash_message'])) {
			$messageData = $_SESSION['flash_message']; 
			echo '<div class="alert alert-' . htmlspecialchars($messageData['type']) . '" style="padding: 10px; margin-bottom: 15px; border: 1px solid #ccc;">' . htmlspecialchars($messageData['message']) . '</div>';
			unset($_SESSION['flash_message']); 
		} 
	}
}
// --- End Helper Functions ---

if (!isset($_SESSION['general_id'])) {
    set_session_message("Please start by filling out the general information.", "warning");
    redirect_to("general.php");
}
$general_id = $_SESSION['general_id'];
$errors = [];

$years = [2021, 2022, 2023, 2024];
$categories_summary = [
    'total_scope1' => array_fill_keys($years, 0.0), 'total_scope2' => array_fill_keys($years, 0.0),
    'total_scope1_plus_2' => array_fill_keys($years, 0.0), 's1_stationary' => array_fill_keys($years, 0.0),
    's1_mobile' => array_fill_keys($years, 0.0), 's1_fugitive' => array_fill_keys($years, 0.0),
    's2_electricity' => array_fill_keys($years, 0.0), 's2_other_energy' => array_fill_keys($years, 0.0),
];
$breakdown_year = 2024;
$s1_stationary_breakdown = [];
$s1_mobile_breakdown = [];
$s1_fugitive_breakdown = [];

// --- Fetch General Info ---
$general_info_data = null;
$stmt_gi = mysqli_prepare($conn, "SELECT * FROM general_info WHERE id = ?");
if ($stmt_gi) {
    mysqli_stmt_bind_param($stmt_gi, "i", $general_id);
    mysqli_stmt_execute($stmt_gi);
    $result_gi = mysqli_stmt_get_result($stmt_gi);
    $general_info_data = mysqli_fetch_assoc($result_gi);
    mysqli_stmt_close($stmt_gi);
} else {
    $errors[] = "Error preparing general info query: " . mysqli_error($conn);
}

// --- Fetch Scope 1 Main Data ---
$scope1_emission_id = null;
$s1_main_db_totals = null;
$sql_s1_main = "SELECT id, stat_co2_emissions, mobile_co2_emissions, fugitive_co2_emissions, total_scope1_emissions FROM scope1_emissions WHERE general_id = ? LIMIT 1";

// ... (rest of your S1 main fetch, which seems to be working fine to get $scope1_emission_id) ...
$stmt_s1_main = mysqli_prepare($conn, $sql_s1_main);
if (!$stmt_s1_main) {
	$errors[] = "Error preparing S1 main: " . mysqli_error($conn); goto display_page; 
}

mysqli_stmt_bind_param($stmt_s1_main, "i", $general_id);
if (!mysqli_stmt_execute($stmt_s1_main)) {
	$errors[] = "Error executing S1 main: " . mysqli_stmt_error($stmt_s1_main); goto display_page; 
}

$res_s1_main = mysqli_stmt_get_result($stmt_s1_main);
if (!$res_s1_main) {
	$errors[] = "Error getting result S1 main: " . mysqli_stmt_error($stmt_s1_main); goto display_page; 
}
$s1_main_db_totals = mysqli_fetch_assoc($res_s1_main);
mysqli_stmt_close($stmt_s1_main);

if ($s1_main_db_totals) {
    $scope1_emission_id = $s1_main_db_totals['id'];
} else {
	if(!$scope1_emission_id && empty($errors)) {
		// This case might mean Scope 1 was skipped or no data was entered.
		// Totals will naturally remain 0 if $scope1_emission_id is null.
	}
}

// --- Fetch and Aggregate Scope 1 Details for Yearly Breakdown ---
if ($scope1_emission_id) { // Only proceed if we have a valid scope1_emission_id
    // 1a. Stationary Combustion Details
    $stmt_stat = mysqli_prepare($conn, "SELECT fuel_type, input_unit, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM stationary_combustion_details WHERE scope1_emission_id = ?");
	if($stmt_stat) {
		mysqli_stmt_bind_param($stmt_stat, "i", $scope1_emission_id);
		mysqli_stmt_execute($stmt_stat);
		$result_stat = mysqli_stmt_get_result($stmt_stat);
		// $stationary_rows_found = 0; // You can remove this counter if not debugging

		while ($db_row_data = mysqli_fetch_assoc($result_stat)) {
			// $stationary_rows_found++;
			$row = $db_row_data; 
			$fuel_type_from_db = $row['fuel_type'];
			$input_unit_from_db = $row['input_unit'];
			$ef = 0; 
			if (isset($emission_factors_data['stationary'][$fuel_type_from_db]) && isset($emission_factors_data['stationary'][$fuel_type_from_db][$input_unit_from_db])) {
				$ef = $emission_factors_data['stationary'][$fuel_type_from_db][$input_unit_from_db];
			} // Removed else with EF NOT FOUND echo for cleaner output once working
			if ($ef > 0) {
				$current_row_for_year_loop = $row; 
				foreach ($years as $year) {
					$qty_col = 'quantity_' . $year;
					$quantity_value = 0;
					if ($current_row_for_year_loop !== null && isset($current_row_for_year_loop[$qty_col])) {
						$quantity_value = floatval($current_row_for_year_loop[$qty_col]);
					}
					if ($quantity_value > 0) {
						$co2e_year = ($quantity_value * $ef) / 1;
						$categories_summary['s1_stationary'][$year] += $co2e_year;
						$categories_summary['total_scope1'][$year] += $co2e_year;
						if ($year == $breakdown_year) {
							$s1_stationary_breakdown[$fuel_type_from_db] = ($s1_stationary_breakdown[$fuel_type_from_db] ?? 0) + $co2e_year;
						}
					}
				}
			}
		}
		mysqli_stmt_close($stmt_stat);
	} else { 
		$errors[] = "Error preparing S1 Stationary details: ".mysqli_error($conn);
	}

    // ---------------------------------------------------------------------------------
    // RE-INSERT MOBILE COMBUSTION DETAILS PROCESSING LOOP HERE
    // ---------------------------------------------------------------------------------
    // 1b. Mobile Combustion Details
    $stmt_mob = mysqli_prepare($conn, "SELECT fuel_type, input_unit, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM mobile_combustion_details WHERE scope1_emission_id = ?");
    if($stmt_mob){
        mysqli_stmt_bind_param($stmt_mob, "i", $scope1_emission_id);
        mysqli_stmt_execute($stmt_mob);
        $result_mob = mysqli_stmt_get_result($stmt_mob);
        while ($db_row_data_mob = mysqli_fetch_assoc($result_mob)) {
            $row_mob = $db_row_data_mob;
            $fuel_type_from_db_mob = $row_mob['fuel_type'];
            $input_unit_from_db_mob = $row_mob['input_unit'];
            $ef_mob = 0;
            if (isset($emission_factors_data['mobile'][$fuel_type_from_db_mob]) && isset($emission_factors_data['mobile'][$fuel_type_from_db_mob][$input_unit_from_db_mob])) {
                $ef_mob = $emission_factors_data['mobile'][$fuel_type_from_db_mob][$input_unit_from_db_mob];
            } // Removed EF NOT FOUND echo
            if ($ef_mob > 0) {
                $current_row_for_year_loop_mob = $row_mob;
                foreach ($years as $year) {
                    $qty_col = 'quantity_' . $year;
                    $quantity_value_mob = 0;
                    if ($current_row_for_year_loop_mob !== null && isset($current_row_for_year_loop_mob[$qty_col])) {
                        $quantity_value_mob = floatval($current_row_for_year_loop_mob[$qty_col]);
                    }
                    if ($quantity_value_mob > 0) {
                        $co2e_year_mob = ($quantity_value_mob * $ef_mob) / 1;
                        $categories_summary['s1_mobile'][$year] += $co2e_year_mob;
                        $categories_summary['total_scope1'][$year] += $co2e_year_mob;
                        if ($year == $breakdown_year) {
                            $s1_mobile_breakdown[$fuel_type_from_db_mob] = ($s1_mobile_breakdown[$fuel_type_from_db_mob] ?? 0) + $co2e_year_mob;
                        }
                    }
                }
            }
        }
        mysqli_stmt_close($stmt_mob);
    } else { 
		$errors[] = "Error preparing S1 Mobile details: ".mysqli_error($conn); 
	}

    // ---------------------------------------------------------------------------------
    // RE-INSERT FUGITIVE EMISSIONS DETAILS PROCESSING LOOP HERE
    // ---------------------------------------------------------------------------------
    // 1c. Fugitive Emissions Details
    $stmt_fug = mysqli_prepare($conn, "SELECT gas_type, input_unit, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM fugitive_emission_details WHERE scope1_emission_id = ?");
    if($stmt_fug){
        mysqli_stmt_bind_param($stmt_fug, "i", $scope1_emission_id);
        mysqli_stmt_execute($stmt_fug);
        $result_fug = mysqli_stmt_get_result($stmt_fug);
        while ($db_row_data_fug = mysqli_fetch_assoc($result_fug)) {
            $row_fug = $db_row_data_fug;
            $gas_type_from_db_fug = $row_fug['gas_type'];
            $input_unit_from_db_fug = $row_fug['input_unit'];
            $gwp_fug = 0;
            if (isset($emission_factors_data['fugitive'][$gas_type_from_db_fug]) && isset($emission_factors_data['fugitive'][$gas_type_from_db_fug][$input_unit_from_db_fug])) {
                $gwp_fug = $emission_factors_data['fugitive'][$gas_type_from_db_fug][$input_unit_from_db_fug];
            } elseif (isset($emission_factors_data['fugitive'][$gas_type_from_db_fug]['kg'])) { // Fallback to 'kg' key for GWP
                $gwp_fug = $emission_factors_data['fugitive'][$gas_type_from_db_fug]['kg'];
            } // Removed GWP NOT FOUND echo
            if ($gwp_fug > 0) {
                $current_row_for_year_loop_fug = $row_fug;
                foreach ($years as $year) {
                    $qty_col = 'quantity_' . $year;
                    $quantity_value_fug_original_unit = 0;
                    if ($current_row_for_year_loop_fug !== null && isset($current_row_for_year_loop_fug[$qty_col])) {
                        $quantity_value_fug_original_unit = floatval($current_row_for_year_loop_fug[$qty_col]);
                    }
                    
                    $qty_in_tonnes_fug = 0;
                    if ($input_unit_from_db_fug === 'kg') {
                        $qty_in_tonnes_fug = $quantity_value_fug_original_unit / 1;
                    } elseif ($input_unit_from_db_fug === 'metric tonnes') {
                        $qty_in_tonnes_fug = $quantity_value_fug_original_unit;
                    } // Else, unit is not suitable for GWP, qty_in_tonnes_fug remains 0

                    if ($qty_in_tonnes_fug > 0) {
                        $co2e_year_fug = $qty_in_tonnes_fug * $gwp_fug;
                        $categories_summary['s1_fugitive'][$year] += $co2e_year_fug;
                        $categories_summary['total_scope1'][$year] += $co2e_year_fug;
                        if ($year == $breakdown_year) {
                            $s1_fugitive_breakdown[$gas_type_from_db_fug] = ($s1_fugitive_breakdown[$gas_type_from_db_fug] ?? 0) + $co2e_year_fug;
                        }
                    }
                }
            }
        }
        mysqli_stmt_close($stmt_fug);
    } else {
		$errors[] = "Error preparing S1 Fugitive details: ".mysqli_error($conn); 
	}
} else { // This 'else' corresponds to `if ($scope1_emission_id)`
	// echo "DEBUG: No scope1_emission_id found, skipping ALL Scope 1 detail processing.<br>";
}

// --- Fetch Scope 2 Main Data ---
$scope2_emission_id = null;
$s2_main_db_totals = null;
$sql_s2_main = "SELECT id, electricity_total_co2e, steam_heat_cooling_total_co2e, include_steam_heat_cooling, total_scope2_emissions FROM scope2_emissions WHERE general_id = ? LIMIT 1";
$stmt_s2_main = mysqli_prepare($conn, $sql_s2_main);

if (!$stmt_s2_main) {
	$errors[] = "Error preparing S2 main: " . mysqli_error($conn); goto display_page; 
}
mysqli_stmt_bind_param($stmt_s2_main, "i", $general_id);

if (!mysqli_stmt_execute($stmt_s2_main)) {
	$errors[] = "Error executing S2 main: " . mysqli_stmt_error($stmt_s2_main); goto display_page;
}
$res_s2_main = mysqli_stmt_get_result($stmt_s2_main);

if (!$res_s2_main) {
	$errors[] = "Error getting result S2 main: " . mysqli_stmt_error($stmt_s2_main); goto display_page;
}
$s2_main_db_totals = mysqli_fetch_assoc($res_s2_main);
mysqli_stmt_close($stmt_s2_main);

if ($s2_main_db_totals) {
    $scope2_emission_id = $s2_main_db_totals['id'];
} else {
    //  if(!$scope2_emission_id && empty($errors)) {
    //     // $errors[] = "No Scope 2 summary data found";
    //  }
}

// --- Fetch and Aggregate Scope 2 Details for Yearly Breakdown ---
if ($scope2_emission_id) {
    // Electricity
    $stmt_elec = mysqli_prepare($conn, "SELECT region, input_unit, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM electricity_consumption_details WHERE scope2_emission_id = ?");
    if($stmt_elec){
        mysqli_stmt_bind_param($stmt_elec, "i", $scope2_emission_id); 
        mysqli_stmt_execute($stmt_elec);
        $result_elec = mysqli_stmt_get_result($stmt_elec);
        
        while ($row = mysqli_fetch_assoc($result_elec)) { // $row contains data for the current electricity entry
            // $raw_s2_electricity[] = $row; // If you also need to store raw data for PDF display

            // Get region and input_unit FROM THE CURRENT $row
            $current_row_region = $row['region'];
            $current_row_input_unit = $row['input_unit'];
			$year_column_keys = ['col1', 'col2', 'col3', 'col4']; 

            // Iterate through each year to apply the year-specific EF
            foreach (array_keys($years) as $year_index) { // $year_index will be 0, 1, 2, 3
                $current_year = $years[$year_index]; // Get 2021, 2022, etc.
                $ef_key_for_year = $year_column_keys[$year_index]; // Get col1, col2, etc.

                $qty_col = 'quantity_' . $current_year;
                $quantity_value = floatval($row[$qty_col] ?? 0);

                if ($quantity_value > 0) {
                    $ef_for_year = 0;
                    if (isset($emission_factors_data_scope2['electricity'][$current_row_region]) && isset($emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit]) && isset($emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit][$ef_key_for_year])) {
                        $ef_for_year = $emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit][$ef_key_for_year];
                    }
                    
                    if ($ef_for_year > 0) {
                        $co2e_year = ($quantity_value * $ef_for_year);
                        // Use $current_year to store in $categories_summary
                        $categories_summary['s2_electricity'][$current_year] += $co2e_year;
                        $categories_summary['total_scope2'][$current_year] += $co2e_year;
                    }
                }
            }
		} 
		mysqli_stmt_close($stmt_elec);
    } else {
        error_log("PDF Download - Error preparing S2 Electricity details: ".mysqli_error($conn));
	}

    // Steam, Heat & Cooling
    if ($s2_main_db_totals && $s2_main_db_totals['include_steam_heat_cooling']) {
        // ... (your SHC details loop - seems okay, ensure EF lookup is robust) ...
        $stmt_shc = mysqli_prepare($conn, "SELECT energy_type, quantity_2021, quantity_2022, quantity_2023, quantity_2024 FROM steam_heat_cooling_details WHERE scope2_emission_id = ?");
        if($stmt_shc){
            mysqli_stmt_bind_param($stmt_shc, "i", $scope2_emission_id); mysqli_stmt_execute($stmt_shc);
            $result_shc = mysqli_stmt_get_result($stmt_shc);
            while ($row = mysqli_fetch_assoc($result_shc)) {
                $ef = $emission_factors_data_scope2['steam_heat_cooling'][$row['energy_type']] ?? 0;
                if ($ef > 0) { 
					foreach ($years as $year) { 
						$qty_col = 'quantity_' . $year; $co2e_year = (floatval($row[$qty_col] ?? 0) * $ef) / 1; 
						$categories_summary['s2_other_energy'][$year] += $co2e_year; $categories_summary['total_scope2'][$year] += $co2e_year; 
					} 
				}
            } mysqli_stmt_close($stmt_shc);
        } else { 
			$errors[] = "Error preparing S2 SHC details: ".mysqli_error($conn);
		}
    }
}

// Calculate Total Scope 1 + 2
foreach ($years as $year) {
    $categories_summary['total_scope1_plus_2'][$year] = ($categories_summary['total_scope1'][$year] ?? 0.0) + ($categories_summary['total_scope2'][$year] ?? 0.0);
}

// Prepare data for Charts (ensure these variables are always defined, even if empty arrays, before json_encode)
$line_chart_labels = json_encode($years);
$line_chart_scope1_data = json_encode(array_values($categories_summary['total_scope1']));
$line_chart_scope2_data = json_encode(array_values($categories_summary['total_scope2']));
$line_chart_total_data = json_encode(array_values($categories_summary['total_scope1_plus_2']));
$s1_stat_pie_labels = json_encode(array_values(array_keys($s1_stationary_breakdown))); // Use array_values for safety if keys are numeric
$s1_stat_pie_data = json_encode(array_values($s1_stationary_breakdown));
$s1_mob_pie_labels = json_encode(array_values(array_keys($s1_mobile_breakdown)));
$s1_mob_pie_data = json_encode(array_values($s1_mobile_breakdown));
$s1_fug_pie_labels = json_encode(array_values(array_keys($s1_fugitive_breakdown)));
$s1_fug_pie_data = json_encode(array_values($s1_fugitive_breakdown));

display_page: // Label for goto if major DB errors occur before HTML rendering
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>GHG Calculator - Result Summary</title>
    <link rel="stylesheet" href="assets/css/main.css" />
	<script type="text/javascript" src="assets/js/hamburger.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
	<script>
        const phpChartData = {
			line: { // Create a 'line' object
				labels: <?php echo $line_chart_labels; ?>,
				scope1: <?php echo $line_chart_scope1_data; ?>,
				scope2: <?php echo $line_chart_scope2_data; ?>,
				total: <?php echo $line_chart_total_data; ?>
			},
			pie_stationary: { // Create a 'pie_stationary' object
				labels: <?php echo $s1_stat_pie_labels; ?>,
				data: <?php echo $s1_stat_pie_data; ?>
			},
			pie_mobile: { // Create a 'pie_mobile' object
				labels: <?php echo $s1_mob_pie_labels; ?>,
				data: <?php echo $s1_mob_pie_data; ?>
			},
			pie_fugitive: { // Create a 'pie_fugitive' object
				labels: <?php echo $s1_fug_pie_labels; ?>,
				data: <?php echo $s1_fug_pie_data; ?>
			},
			breakdown_year: <?php echo json_encode($breakdown_year); ?>
		};
		const phpErrorsExist = <?php echo !empty($errors) ? 'true' : 'false'; ?>;
    </script>
    <style>
		table th {color: #000}
        .summary-table { width: 100%; border-collapse: collapse; margin-bottom: 30px; font-size: 0.9em; }
        .summary-table th, .summary-table td { border: 1px solid #ccc; padding: 6px; text-align: left; }
        .summary-table th { background-color: #f0f0f0; }
        .summary-table td.year-data { text-align: right; }
        .chart-container { width: 48%; margin-bottom: 20px; display: inline-block; vertical-align: top; border:1px solid #eee; padding:10px; box-sizing: border-box; min-height: 300px;}
        .full-width-chart-container { width: 98%; margin-bottom: 20px; border:1px solid #eee; padding:10px; min-height: 350px; }
        @media (max-width: 900px) { .chart-container { width: 98%; } }
        .section-title { font-size: 1.4em; margin-top: 25px; margin-bottom: 15px; border-bottom: 2px solid #eee; padding-bottom: 8px;}
        .errors { color:red; border:1px solid red; padding:10px; margin-bottom:15px; }
        .errors ul { padding-left: 20px; margin:0; }
    </style>
</head>
<body>
    <div id="page-wrapper">
        <header id="header">
			<div id="logo">
				<a href="index.html"><img src="images/cmm-logo.png" alt=""/></a>
			</div>
			<nav id="nav">
				<ul>
					<li><a href="index.html">Home</a></li>
					<li><a href="methodology.html">Methodology</a>
						<ul>
							<li><a href="#">User Guide</a></li>
						</ul>
					</li>
					<li><a href="about.html">About</a></li>
					<li><a href="general.php" class="button">Get Started</a></li>
				</ul>
			</nav>
		</header>
        <section id="summary-page" style="padding: 20px;">
            <div class="wrapper">
                <div class="container" style="max-width: 1200px; margin: auto;">
                    <div id="formbox" class="general">
						<div id="breadcrumb">
							<ul class="breadcrumb">
								<li><a>General</a></li>
								<li><a>Scope 1</a></li>
								<li><a>Scope 2</a></li>
								<li><a>Intensity</a></li>
								<li class="active"><a>Result Summary</a></li>
							</ul>
						</div>
                        
						<div id="hiddenChartContainer" style="position: absolute; left: -9999px; top: -9999px; width: 800px; height: 1800px; /* height to accommodate all canvases stacked */ overflow: hidden;">
							<canvas id="ghgByScopeChart_forPDF" width="800" height="400"></canvas>
							<canvas id="stationaryBreakdownChart_forPDF" width="400" height="400"></canvas>
							<canvas id="mobileBreakdownChart_forPDF" width="400" height="400"></canvas>
							<canvas id="fugitiveBreakdownChart_forPDF" width="400" height="400"></canvas>
						</div>
						
						<div class="note align-left"><div class="midtitle">Result Summary </div></div>
                        
                        <?php if (!empty($errors)): ?>
                            <div class="errors">
                                <strong>Error retrieving summary data:</strong>
                                <ul><?php foreach ($errors as $error): echo "<li>".htmlspecialchars($error)."</li>"; endforeach; ?></ul>
                            </div>
                        <?php else: // Only show tables and charts if no major errors ?>
						
						<div class="box" style="text-align: justify">
							<p>Your Scope 1 and Scope 2 GHG emissions report, based on the data provided, is available for download in two formats: Excel and PDF.</p>
							<p>The Excel report is structured as follows:</p>
							<ul>
								<li>Sheet 1: Input from activity data</li>
								<li>Sheet 2: Corresponding GHG emission results</li>
							</ul>
							<p>This format is designed for simplicity and ease of integration into internal accounting systems or external reporting platforms.</p>
							<p>The PDF report provides a more user-friendly summary, including:</p>
							<ul>
								<li>Visual charts illustrating emissions trends over time</li>
								<li>Identification of key emission sources within your operations</li>
							</ul>
							<p>For any questions about the calculator, please contact us at <a href="mailto:general@capitalmarketsmalaysia.com">general@capitalmarketsmalaysia.com</a>.</p>
							
							<div id="action-btn">
								<a href="download_excel.php"><button class="excel-download">Download Excel Data</button></a>
								<a href="download_pdf.php"><button id="downloadPdfBtn" class="pdf-download">Download PDF Report</button></a>
								<!-- Add download/print buttons here later -->
							</div><!--end action btn-->
                        <?php endif; // End of error check for displaying tables/charts ?>
						</div>
                    </div>
                </div>
            </div>
        </section>
        <footer id="footer">Copyright © 2025 Capital Markets Malaysia. All Rights Reserved.</footer>
    </div>
<script>
document.addEventListener('DOMContentLoaded', function () {
    // This phpChartData object is defined by PHP json_encode just before this script block
    // Ensure it has the structure:
    // const phpChartData = {
    //     line: { labels: [...], scope1: [...], scope2: [...], total: [...] },
    //     pie_stationary: { labels: [...], data: [...] },
    //     pie_mobile: { labels: [...], data: [...] },
    //     pie_fugitive: { labels: [...], data: [...] },
    //     breakdown_year: "2024" // or the actual year
    // };
    // const phpErrorsExist = true/false;

    if (typeof phpErrorsExist !== 'undefined' && phpErrorsExist) {
        console.warn("PHP errors detected on page load. Chart initialization and PDF download might be affected.");
        const pdfBtn = document.getElementById('downloadPdfBtn');
        if(pdfBtn) {
            pdfBtn.title = "Data errors prevent PDF generation.";
            // pdfBtn.disabled = true; // Keep it enabled, let user try, JS will alert
            // pdfBtn.style.opacity = 0.6;
        }
        // We don't return here, let user attempt download and see JS alerts if data is bad for charts
    }

    if (typeof phpChartData === 'undefined') {
        console.error("FATAL ERROR: phpChartData JavaScript object is not defined. Charts and PDF download will not work.");
        const pdfBtn = document.getElementById('downloadPdfBtn');
        if(pdfBtn) {  pdfBtn.disabled = true; pdfBtn.textContent = "Chart Data Error"; }
        return;
    }

    function getPieColors(count) {
        const baseColors = ['rgba(255, 99, 132, 0.8)','rgba(54, 162, 235, 0.8)','rgba(255, 206, 86, 0.8)','rgba(75, 192, 192, 0.8)','rgba(153, 102, 255, 0.8)','rgba(255, 159, 64, 0.8)'];
        let colors = []; for(let i=0;i<count;i++) colors.push(baseColors[i % baseColors.length]); return colors;
    }

    // --- VISIBLE CHARTS INITIALIZATION ---
    // These use the *visible* canvas IDs from your HTML.
    // They will only render if the corresponding data in phpChartData is valid.

    try { // Wrap chart initializations in a try-catch
        const ctxVisibleGhgByScope = document.getElementById('ghgByScopeChart');
        if (ctxVisibleGhgByScope && phpChartData.line && phpChartData.line.labels && phpChartData.line.labels.length > 0) {
            new Chart(ctxVisibleGhgByScope.getContext('2d'), {
                type: 'line',
                data: { labels: phpChartData.line.labels, datasets: [
                        { label: 'Total Scope 1 (tCO2e)', data: phpChartData.line.scope1, borderColor: 'rgb(54, 162, 235)', tension: 0.1, fill: false },
                        { label: 'Total Scope 2 (tCO2e)', data: phpChartData.line.scope2, borderColor: 'rgb(255, 159, 64)', tension: 0.1, fill: false },
                        { label: 'Total Scope 1+2 (tCO2e)', data: phpChartData.line.total, borderColor: 'rgb(75, 192, 192)', tension: 0.1, fill: false }
                    ]
                }, options: { responsive: true, plugins: { title: { display: true, text: 'GHG Emissions by Scope in metric tonnes CO2e' } }, scales: { y: { beginAtZero: true, title: { display: true, text: 'GHG Emissions (metric tons of CO2e)' } } } }
            });
        } else if(ctxVisibleGhgByScope) {
             ctxVisibleGhgByScope.parentElement.innerHTML = '<p style="text-align:center; padding-top:50px;">No data for GHG by Scope chart.</p>';
        }

        const ctxVisibleStationary = document.getElementById('stationaryBreakdownChart');
        if (ctxVisibleStationary && phpChartData.pie_stationary && phpChartData.pie_stationary.labels && phpChartData.pie_stationary.labels.length > 0 && phpChartData.pie_stationary.data.some(d => d > 0)) {
            new Chart(ctxVisibleStationary.getContext('2d'), {
                type: 'pie',
                data: { labels: phpChartData.pie_stationary.labels, datasets: [{ data: phpChartData.pie_stationary.data, backgroundColor: getPieColors(phpChartData.pie_stationary.labels.length), hoverOffset: 4 }] },
                options: { responsive: true, maintainAspectRatio: false, plugins: { title: { display: true, text: (phpChartData.breakdown_year || 'Year') + ' Stationary Breakdown (tCO2e)' } } }
            });
        } else if(ctxVisibleStationary) {
            ctxVisibleStationary.parentElement.innerHTML = '<p style="text-align:center; padding-top:50px;">No stationary data for ' + (phpChartData.breakdown_year || 'selected year') + ' to display chart.</p>';
        }

        const ctxVisibleMobile = document.getElementById('mobileBreakdownChart');
        if (ctxVisibleMobile && phpChartData.pie_mobile && phpChartData.pie_mobile.labels && phpChartData.pie_mobile.labels.length > 0 && phpChartData.pie_mobile.data.some(d => d > 0)) {
            new Chart(ctxVisibleMobile.getContext('2d'), {
                type: 'pie',
                data: { labels: phpChartData.pie_mobile.labels, datasets: [{ data: phpChartData.pie_mobile.data, backgroundColor: getPieColors(phpChartData.pie_mobile.labels.length), hoverOffset: 4 }] },
                options: { responsive: true, maintainAspectRatio: false, plugins: { title: { display: true, text: (phpChartData.breakdown_year || 'Year') + ' Mobile Breakdown (tCO2e)' } } }
            });
        } else if(ctxVisibleMobile) {
            ctxVisibleMobile.parentElement.innerHTML = '<p style="text-align:center; padding-top:50px;">No mobile data for ' + (phpChartData.breakdown_year || 'selected year') + ' to display chart.</p>';
        }
        
        // Note: Your HTML for output.php (code.txt) was missing the canvas for fugitiveBreakdownChart in the visible section
        // It was only in hiddenChartContainer. If you want it visible, add:
        // <div class="chart-container"><canvas id="fugitiveBreakdownChart"></canvas></div>
        const ctxVisibleFugitive = document.getElementById('fugitiveBreakdownChart'); // Assumes visible canvas exists
         if (ctxVisibleFugitive && phpChartData.pie_fugitive && phpChartData.pie_fugitive.labels && phpChartData.pie_fugitive.labels.length > 0 && phpChartData.pie_fugitive.data.some(d => d > 0)) {
            new Chart(ctxVisibleFugitive.getContext('2d'), {
                type: 'pie',
                data: { labels: phpChartData.pie_fugitive.labels, datasets: [{ data: phpChartData.pie_fugitive.data, backgroundColor: getPieColors(phpChartData.pie_fugitive.labels.length), hoverOffset: 4 }] },
                options: { responsive: true, maintainAspectRatio: false, plugins: { title: { display: true, text: (phpChartData.breakdown_year || 'Year') + ' Fugitive Breakdown (tCO2e)' } } }
            });
        } else if(ctxVisibleFugitive) {
            ctxVisibleFugitive.parentElement.innerHTML = '<p style="text-align:center; padding-top:50px;">No fugitive data for ' + (phpChartData.breakdown_year || 'selected year') + ' to display chart.</p>';
        }

    } catch (chartInitError) {
        console.error("Error initializing visible charts:", chartInitError);
    }

    // --- PDF DOWNLOAD BUTTON LOGIC ---
    const downloadPdfButton = document.getElementById('downloadPdfBtn'); 
    if (downloadPdfButton) {
        downloadPdfButton.addEventListener('click', function() {
            // ... (The PDF download logic from your code.txt, which uses hidden canvases
            //      'ghgByScopeChart_forPDF', 'stationaryBreakdownChart_forPDF', etc.
            //      and phpChartData) ...
            // This part of your code seemed mostly correct in its structure for generating images
            // The main change is to ensure it references phpChartData correctly.

            const thisButton = this;
            thisButton.disabled = true;
            thisButton.textContent = 'Preparing PDF...';
            const clientTimestamp = new Date().getTime();
            
            const chartsToRenderAndSave = [
                { canvasId: 'ghgByScopeChart_forPDF', type: 'line', chartDataKey: 'line', options: { responsive: false, animation: { duration: 0 }, width:800, height:400, plugins: { title: { display: true, text: 'GHG Emissions by Scope (tCO2e)' } }, scales: { y: { beginAtZero: true } } }, baseFilename: 'chart_ghg_by_scope.png' },
                { canvasId: 'stationaryBreakdownChart_forPDF', type: 'pie', chartDataKey: 'pie_stationary', options: { responsive: false, animation: { duration: 0 }, width:400, height:400, plugins: { title: { display: true, text: (phpChartData.breakdown_year || 'Year') + ' Stationary Breakdown (tCO2e)' } } }, baseFilename: 'chart_stationary_breakdown.png' },
                { canvasId: 'mobileBreakdownChart_forPDF', type: 'pie', chartDataKey: 'pie_mobile', options: { responsive: false, animation: { duration: 0 }, width:400, height:400, plugins: { title: { display: true, text: (phpChartData.breakdown_year || 'Year') + ' Mobile Breakdown (tCO2e)' } } }, baseFilename: 'chart_mobile_breakdown.png' },
                { canvasId: 'fugitiveBreakdownChart_forPDF', type: 'pie', chartDataKey: 'pie_fugitive', options: { responsive: false, animation: { duration: 0 }, width:400, height:400, plugins: { title: { display: true, text: (phpChartData.breakdown_year || 'Year') + ' Fugitive Breakdown (tCO2e)' } } }, baseFilename: 'chart_fugitive_breakdown.png' }
            ];

            let promises = [];
            let timestampedFilenamesForPdf = {};
            const chartNameMapping = {
				'ghgByScopeChart_forPDF': 'chart_ghg_by_scope', // <<< Key here MUST match canvasId above
				'stationaryBreakdownChart_forPDF': 'chart_stationary_breakdown',
				'mobileBreakdownChart_forPDF': 'chart_mobile_breakdown',
				'fugitiveBreakdownChart_forPDF': 'chart_fugitive_breakdown'
			};
            let chartsProcessedCount = 0;

            chartsToRenderAndSave.forEach(chartInfo => {
                const canvas = document.getElementById(chartInfo.canvasId);
                const currentChartDataForPdf = phpChartData[chartInfo.chartDataKey]; // Use the grouped data

                if (!canvas) {
                    console.warn('Canvas for PDF not found:', chartInfo.canvasId);
                    promises.push(Promise.resolve({ success: false, message: 'Canvas not found: ' + chartInfo.canvasId, filename: chartInfo.baseFilename }));
                    chartsProcessedCount++;
                    if (chartsProcessedCount === chartsToRenderAndSave.length) checkAllChartsProcessedPdf();
                    return;
                }
                if (!currentChartDataForPdf) {
                    console.warn('Chart data not found in phpChartData for key:', chartInfo.chartDataKey, 'for canvas:', chartInfo.canvasId);
                    promises.push(Promise.resolve({ success: false, message: 'Chart data undefined for ' + chartInfo.canvasId, filename: chartInfo.baseFilename }));
                    chartsProcessedCount++;
                    if (chartsProcessedCount === chartsToRenderAndSave.length) checkAllChartsProcessedPdf();
                    return;
                }

                const ctx = canvas.getContext('2d');
                let chartDatasetsForPdf;

                if (chartInfo.type === 'line') {
                    chartDatasetsForPdf = [
                        { label: 'Total Scope 1', data: currentChartDataForPdf.scope1, borderColor: 'rgb(54, 162, 235)', fill: false },
                        { label: 'Total Scope 2', data: currentChartDataForPdf.scope2, borderColor: 'rgb(255, 159, 64)', fill: false },
                        { label: 'Total Scope 1+2', data: currentChartDataForPdf.total, borderColor: 'rgb(75, 192, 192)', fill: false }
                    ];
                } else if (chartInfo.type === 'pie') {
                    if (!currentChartDataForPdf.labels || currentChartDataForPdf.labels.length === 0 || !currentChartDataForPdf.data || !currentChartDataForPdf.data.some(d => d > 0)) {
                        console.warn('No data for PDF pie chart:', chartInfo.canvasId);
                        promises.push(Promise.resolve({ success: true, message: 'No data, skipped: ' + chartInfo.canvasId, filename: chartInfo.baseFilename, noData: true }));
                        chartsProcessedCount++;
                        if (chartsProcessedCount === chartsToRenderAndSave.length) checkAllChartsProcessedPdf();
                        return;
                    }
                    chartDatasetsForPdf = [{ data: currentChartDataForPdf.data, backgroundColor: getPieColors(currentChartDataForPdf.labels.length) }];
                }

                if (Chart.getChart(canvas)) Chart.getChart(canvas).destroy();
                const chartInstance = new Chart(ctx, { type: chartInfo.type, data: { labels: currentChartDataForPdf.labels, datasets: chartDatasetsForPdf }, options: chartInfo.options });

                const chartPromise = new Promise((resolve, reject) => {
                    requestAnimationFrame(() => {
                        try {
                            const imageDataUrl = chartInstance.toBase64Image();
                            const dynamicFilename = clientTimestamp + '_' + chartInfo.baseFilename;
                            
                            // Declare paramKey here, only once within this callback scope
                            const paramKey = chartNameMapping[chartInfo.canvasId]; 
                            // ^^^ Ensure this is the ONLY 'const paramKey' or 'let paramKey' for this variable
                            //     within this specific requestAnimationFrame callback.

                            console.log("Processing chart for PDF. canvasId:", chartInfo.canvasId, "Mapped paramKey:", paramKey); // Combined debug

                            if (paramKey) {
                                timestampedFilenamesForPdf[paramKey] = dynamicFilename;
                                console.log("Stored filename for PDF:", paramKey, "=", dynamicFilename);
                            } else {
                                console.error("CRITICAL: paramKey is UNDEFINED or FALSY for canvasId:", chartInfo.canvasId, ". Check chartNameMapping keys. chartNameMapping content:", JSON.stringify(chartNameMapping));
                            }

                            // Only proceed with fetch if paramKey was valid, to avoid issues in download_pdf.php
                            if (paramKey) {
                                fetch('save_chart_image.php', {
                                    method: 'POST',
                                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                                    body: 'imageData=' + encodeURIComponent(imageDataUrl) + '&filename=' + encodeURIComponent(dynamicFilename)
                                })
                                .then(response => response.json())
                                .then(data => {
                                    if (!data.success) {
                                        console.error('Failed to save chart (server):', dynamicFilename, 'Error:', data.message);
                                    }
                                    resolve(data); 
                                })
                                .catch(err => {
                                    console.error('Fetch error for saving chart:', dynamicFilename, err);
                                    reject({ success: false, message: 'Fetch error: ' + err.message, filename: dynamicFilename });
                                })
                                .finally(() => {
                                    chartInstance.destroy();
                                    chartsProcessedCount++;
                                    if (chartsProcessedCount === chartsToRenderAndSave.length) {
                                        checkAllChartsProcessedPdf();
                                    }
                                });
                            } else {
                                // If paramKey is not found, we can't properly link this chart image for the PDF
                                // We should treat this as a failure for this specific chart.
                                console.error("Skipping fetch for chart due to missing paramKey:", chartInfo.canvasId);
                                chartInstance.destroy();
                                chartsProcessedCount++;
                                if (chartsProcessedCount === chartsToRenderAndSave.length) {
                                    checkAllChartsProcessedPdf();
                                }
                                reject({ success: false, message: 'paramKey undefined for ' + chartInfo.canvasId, filename: chartInfo.baseFilename });
                            }
                        } catch (e) {
                            console.error("Error in requestAnimationFrame for", chartInfo.canvasId, e);
                            if (chartInstance && typeof chartInstance.destroy === 'function') { // Ensure chartInstance exists before destroying
                                chartInstance.destroy();
                            }
                            chartsProcessedCount++;
                            if (chartsProcessedCount === chartsToRenderAndSave.length) {
                                checkAllChartsProcessedPdf();
                            }
                            reject({ success: false, message: 'Error generating/processing image: '.concat(e.message), filename: chartInfo.baseFilename });
                        }
                    }); // End requestAnimationFrame
                });
                promises.push(chartPromise);
            });

            function checkAllChartsProcessedPdf() { // Renamed to avoid conflict
                Promise.all(promises).then(results => {
                    let allSaved = results.every(res => res.success === true);
                    let attemptedChartsWithData = chartsToRenderAndSave.filter(cDef => {
                        const dataToCheck = phpChartData[cDef.chartDataKey];
                        if (cDef.type === 'pie') return dataToCheck.labels && dataToCheck.labels.length > 0 && dataToCheck.data && dataToCheck.data.some(d => d > 0);
                        return dataToCheck.labels && dataToCheck.labels.length > 0; // For line charts too
                    });
                    let allNecessaryFilenamesGenerated = Object.keys(timestampedFilenamesForPdf).length === attemptedChartsWithData.length;

                    if (allSaved && allNecessaryFilenamesGenerated) {
                        const params = new URLSearchParams();
                        for (const key in timestampedFilenamesForPdf) { params.append(key, timestampedFilenamesForPdf[key]); }
                        window.location.href = 'download_pdf.php?' + params.toString();
                    } else {
                        alert('Error saving one or more chart images for PDF. Console has details.');
                        console.error('PDF Chart saving results:', results, 'Filenames:', timestampedFilenamesForPdf);
                    }
                }).catch(error => {
                    alert('Critical error processing chart images for PDF.');
                    console.error('PDF Promise.all catch:', error);
                }).finally(() => {
                    thisButton.disabled = false; thisButton.textContent = 'Download PDF Report';
                });
            }
            if (chartsToRenderAndSave.length === 0 || promises.length === 0 && chartsProcessedCount === chartsToRenderAndSave.length) {
                 checkAllChartsProcessedPdf();
            }
        });
    } // End if (downloadPdfButton)
});
</script>
<script src="assets/js/jquery.scrolly.min.js"></script>
<script src="assets/js/jquery.dropotron.min.js"></script>
<script src="assets/js/jquery.scrollex.min.js"></script>
<script src="assets/js/skel.min.js"></script>
<script src="assets/js/util.js"></script>
<!--[if lte IE 8]><script src="assets/js/ie/respond.min.js"></script><![endif]-->
<script src="assets/js/main.js"></script>
</body>
</html>