<?php
// admin/view_submissions.php
session_start();
require_once 'includes/auth_check.php'; // Protects the page
require_once '../config.php';             // For DB connection $conn
require_once '../includes/emission_factors_config.php'; // CRITICAL for EF data

$page_title = "View GHG Submission Details";
$errors = [];

// Initialize arrays for summary data
$years = [2021, 2022, 2023, 2024];
$categories_summary = [
    'total_scope1' => array_fill_keys($years, 0.0),
	'total_scope2' => array_fill_keys($years, 0.0),
    'total_scope1_plus_2' => array_fill_keys($years, 0.0),
	's1_stationary' => array_fill_keys($years, 0.0),
    's1_mobile' => array_fill_keys($years, 0.0),
	's1_fugitive' => array_fill_keys($years, 0.0),
    's2_electricity' => array_fill_keys($years, 0.0),
	's2_other_energy' => array_fill_keys($years, 0.0),
];
$breakdown_year_admin = 2024;
$s1_stationary_breakdown_admin = []; $s1_mobile_breakdown_admin = []; $s1_fugitive_breakdown_admin = [];

// Chart data JSON strings - initialize
$line_chart_labels_admin = json_encode($years);
$line_chart_scope1_data_admin = json_encode(array_fill_keys($years, 0.0));
$line_chart_scope2_data_admin = json_encode(array_fill_keys($years, 0.0));
$line_chart_total_data_admin = json_encode(array_fill_keys($years, 0.0));
$s1_stat_pie_labels_admin = json_encode([]); $s1_stat_pie_data_admin = json_encode([]);
$s1_mob_pie_labels_admin = json_encode([]); $s1_mob_pie_data_admin = json_encode([]);
$s1_fug_pie_labels_admin = json_encode([]); $s1_fug_pie_data_admin = json_encode([]);

$submission_details_general = null;
$scope1_main_data_from_db = null; // To store the fetched scope1_emissions record
$scope2_main_data_from_db = null; // To store the fetched scope2_emissions record

// These will hold the RAW fetched details for display
$raw_scope1_stationary_details = [];
$raw_scope1_mobile_details = [];
$raw_scope1_fugitive_details = [];
$raw_scope2_electricity_details = [];
$raw_scope2_shc_details = [];

// --- NEW: Initialize Intensity Metrics Array ---
$intensity_metrics = [
    'scope1_intensity' => array_fill_keys($years, null),
    'scope2_intensity' => array_fill_keys($years, null),
    'total_intensity' => array_fill_keys($years, null),
];
$annual_revenues_by_year = array_fill_keys($years, null); // For storing fetched revenues

$year_to_col_map = [
    2021 => 'col1',
    2022 => 'col2',
    2023 => 'col3',
    2024 => 'col4',
];

$submissions = []; // For the list view

if (isset($_GET['view_id'])) {
    $view_id = intval($_GET['view_id']); // This is the general_id

    if ($view_id > 0) {
        // 1. FETCH GENERAL INFO
        $stmt_gen = mysqli_prepare($conn, "SELECT * FROM general_info WHERE id = ?");
        if ($stmt_gen) {
            mysqli_stmt_bind_param($stmt_gen, "i", $view_id); mysqli_stmt_execute($stmt_gen);
            $res_gen = mysqli_stmt_get_result($stmt_gen);
            $submission_details_general = mysqli_fetch_assoc($res_gen);
            mysqli_stmt_close($stmt_gen);
        } else { 
			$errors[] = "Error preparing general info query: " . mysqli_error($conn); 
		}

        if ($submission_details_general) {
            // 2. FETCH SCOPE 1 MAIN RECORD (to get its ID for linking details AND its stored totals)
            $scope1_emission_id = null;
            $stmt_s1_main = mysqli_prepare($conn, "SELECT * FROM scope1_emissions WHERE general_id = ? LIMIT 1");
            if ($stmt_s1_main) {
                mysqli_stmt_bind_param($stmt_s1_main, "i", $view_id); mysqli_stmt_execute($stmt_s1_main);
                $res_s1_main = mysqli_stmt_get_result($stmt_s1_main);
                $scope1_main_data_from_db = mysqli_fetch_assoc($res_s1_main);
                if ($scope1_main_data_from_db) $scope1_emission_id = $scope1_main_data_from_db['id'];
                mysqli_stmt_close($stmt_s1_main);
            } else { 
				$errors[] = "Error preparing S1 main query: " . mysqli_error($conn); 
			}

            // 3. FETCH SCOPE 1 RAW DETAILS & CALCULATE SUMMARY CATEGORIES
            if ($scope1_emission_id) {
                // Stationary
                $stmt_stat = mysqli_prepare($conn, "SELECT * FROM stationary_combustion_details WHERE scope1_emission_id = ?");
                if($stmt_stat) {
                    mysqli_stmt_bind_param($stmt_stat, "i", $scope1_emission_id); mysqli_stmt_execute($stmt_stat);
                    $result_stat = mysqli_stmt_get_result($stmt_stat);
                    while ($row = mysqli_fetch_assoc($result_stat)) {
                        $raw_scope1_stationary_details[] = $row;
                        $ef = $emission_factors_data['stationary'][$row['fuel_type']][$row['input_unit']] ?? 0;
                        if ($ef > 0) { 
							foreach ($years as $year) { 
								$qty_col = 'quantity_' . $year; 
								$co2e_year = (floatval($row[$qty_col] ?? 0) * $ef); 
								$categories_summary['s1_stationary'][$year] += $co2e_year; 
								$categories_summary['total_scope1'][$year] += $co2e_year; 
								
								if ($year == $breakdown_year_admin) { 
									$s1_stationary_breakdown_admin[$row['fuel_type']] = ($s1_stationary_breakdown_admin[$row['fuel_type']] ?? 0) + $co2e_year; 
								} 
							} 
						}
                    } mysqli_stmt_close($stmt_stat);
                } else { 
					$errors[] = "Error S1 Stationary details query: " . mysqli_error($conn); 
				}

                // Mobile
                $stmt_mob = mysqli_prepare($conn, "SELECT * FROM mobile_combustion_details WHERE scope1_emission_id = ?");
                if($stmt_mob){
                    mysqli_stmt_bind_param($stmt_mob, "i", $scope1_emission_id); mysqli_stmt_execute($stmt_mob);
                    $result_mob = mysqli_stmt_get_result($stmt_mob);
                    while ($row = mysqli_fetch_assoc($result_mob)) {
                        $raw_scope1_mobile_details[] = $row;
                        $ef = $emission_factors_data['mobile'][$row['fuel_type']][$row['input_unit']] ?? 0;
                        if ($ef > 0) { 
							foreach ($years as $year) { 
								$qty_col = 'quantity_' . $year; 
								$co2e_year = (floatval($row[$qty_col] ?? 0) * $ef); 
								$categories_summary['s1_mobile'][$year] += $co2e_year; 
								$categories_summary['total_scope1'][$year] += $co2e_year; 
								
								if ($year == $breakdown_year_admin) {
									$s1_mobile_breakdown_admin[$row['fuel_type']] = ($s1_mobile_breakdown_admin[$row['fuel_type']] ?? 0) + $co2e_year; 
								} 
							} 
						}
                    } mysqli_stmt_close($stmt_mob);
                } else { 
					$errors[] = "Error S1 Mobile details query: " . mysqli_error($conn); 
				}

                // Fugitive
                $stmt_fug = mysqli_prepare($conn, "SELECT * FROM fugitive_emission_details WHERE scope1_emission_id = ?");
                if($stmt_fug){
                    mysqli_stmt_bind_param($stmt_fug, "i", $scope1_emission_id); mysqli_stmt_execute($stmt_fug);
                    $result_fug = mysqli_stmt_get_result($stmt_fug);
                    while ($row = mysqli_fetch_assoc($result_fug)) {
                        $raw_scope1_fugitive_details[] = $row;
                        $gwp = $emission_factors_data['fugitive'][$row['gas_type']][$row['input_unit']] ?? $emission_factors_data['fugitive'][$row['gas_type']]['kg'] ?? 0;
                        if ($gwp > 0) { 
							foreach ($years as $year) { 
								$qty_col = 'quantity_' . $year; 
								$qty_original_unit = floatval($row[$qty_col] ?? 0); 
								$qty_in_tonnes = ($row['input_unit'] === 'kg') ? $qty_original_unit : (($row['input_unit'] === 'metric tonnes') ? $qty_original_unit : 0) ; 
								
								if ($qty_in_tonnes > 0) { 
									$co2e_year = $qty_in_tonnes * $gwp; 
									$categories_summary['s1_fugitive'][$year] += $co2e_year; $categories_summary['total_scope1'][$year] += $co2e_year; 
									if ($year == $breakdown_year_admin) { 
										$s1_fugitive_breakdown_admin[$row['gas_type']] = ($s1_fugitive_breakdown_admin[$row['gas_type']] ?? 0) + $co2e_year; 
									} 
								} 
							} 
						}
                    } mysqli_stmt_close($stmt_fug);
                } else { 
					$errors[] = "Error S1 Fugitive details query: " . mysqli_error($conn); 
				}
            } // End if ($scope1_emission_id)

            // 4. FETCH SCOPE 2 MAIN RECORD
            $scope2_emission_id = null;
            $stmt_s2_main = mysqli_prepare($conn, "SELECT * FROM scope2_emissions WHERE general_id = ? LIMIT 1");
            if ($stmt_s2_main) {
                mysqli_stmt_bind_param($stmt_s2_main, "i", $view_id); mysqli_stmt_execute($stmt_s2_main);
                $res_s2_main = mysqli_stmt_get_result($stmt_s2_main);
                $scope2_main_data_from_db = mysqli_fetch_assoc($res_s2_main);
                if ($scope2_main_data_from_db) $scope2_emission_id = $scope2_main_data_from_db['id'];
                mysqli_stmt_close($stmt_s2_main);
            } else { 
				$errors[] = "Error preparing S2 main query: " . mysqli_error($conn); 
			}
			
			// --- 5. NEW/ENSURE: Fetch Annual Revenue Data ---
            $stmt_revenue = mysqli_prepare($conn, "SELECT revenue_2021, revenue_2022, revenue_2023, revenue_2024 FROM annual_revenues WHERE general_id = ? LIMIT 1");
            if ($stmt_revenue) {
                mysqli_stmt_bind_param($stmt_revenue, "i", $view_id); // Use $view_id which is general_id here
                if (mysqli_stmt_execute($stmt_revenue)) {
                    $result_revenue = mysqli_stmt_get_result($stmt_revenue);
                    if ($revenue_data_row = mysqli_fetch_assoc($result_revenue)) {
                        foreach ($years as $year) {
                            $annual_revenues_by_year[$year] = isset($revenue_data_row['revenue_'.$year]) ? floatval($revenue_data_row['revenue_'.$year]) : null;
                        }
                    } // else { $errors[] = "No annual revenue data found for this submission."; }
                } else { 
					$errors[] = "Error executing revenue query: " . mysqli_stmt_error($stmt_revenue); 
				}
                mysqli_stmt_close($stmt_revenue);
            } else { 
				$errors[] = "Error preparing revenue query: " . mysqli_error($conn); 
			}

            // 6. FETCH SCOPE 2 RAW DETAILS & CALCULATE SUMMARY
            if ($scope2_emission_id) {
                $stmt_elec = mysqli_prepare($conn, "SELECT * FROM electricity_consumption_details WHERE scope2_emission_id = ?");

				if($stmt_elec){
					mysqli_stmt_bind_param($stmt_elec, "i", $scope2_emission_id); 
					mysqli_stmt_execute($stmt_elec);
					$result_elec = mysqli_stmt_get_result($stmt_elec);
					
					while ($row = mysqli_fetch_assoc($result_elec)) {
						$raw_scope2_electricity_details[] = $row;

						$current_row_region = $row['region'] ?? null;
						$current_row_input_unit = $row['input_unit'] ?? null;
						$year_column_keys = ['col1', 'col2', 'col3', 'col4'];

						foreach (array_keys($years) as $year_index) { // $year_index will be 0, 1, 2, 3
							$current_year = $years[$year_index]; // Get 2021, 2022, etc.
							$ef_key_for_year = $year_column_keys[$year_index]; // Get col1, col2, etc.

							$qty_col = 'quantity_' . $current_year;
							$quantity_value = floatval($row[$qty_col] ?? 0);

							if ($quantity_value > 0) {
								$ef_for_year = 0;
								if (isset($emission_factors_data_scope2['electricity'][$current_row_region]) && isset($emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit]) && isset($emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit][$ef_key_for_year])) {
									$ef_for_year = $emission_factors_data_scope2['electricity'][$current_row_region][$current_row_input_unit][$ef_key_for_year];
								}
								
								if ($ef_for_year > 0) {
									$co2e_year = ($quantity_value * $ef_for_year);
									// Use $current_year to store in $categories_summary
									$categories_summary['s2_electricity'][$current_year] += $co2e_year;
									$categories_summary['total_scope2'][$current_year] += $co2e_year;
								}
							}
						}
					} 
					mysqli_stmt_close($stmt_elec);
				} else { 
					$errors[] = "Error preparing S2 Electricity details: ".mysqli_error($conn); 
					error_log("AdminView - Electricity: Error preparing S2 Electricity details: ".mysqli_error($conn)); // LOG
				}

                if ($scope2_main_data_from_db && $scope2_main_data_from_db['include_steam_heat_cooling']) {
                    $stmt_shc = mysqli_prepare($conn, "SELECT * FROM steam_heat_cooling_details WHERE scope2_emission_id = ?");
                     if($stmt_shc){
                        mysqli_stmt_bind_param($stmt_shc, "i", $scope2_emission_id); mysqli_stmt_execute($stmt_shc);
                        $result_shc = mysqli_stmt_get_result($stmt_shc);
                        while ($row = mysqli_fetch_assoc($result_shc)) {
                            $raw_scope2_shc_details[] = $row;
                            $ef = $emission_factors_data_scope2['steam_heat_cooling'][$row['energy_type']] ?? 0;
                            if ($ef > 0) { foreach ($years as $year) { $qty_col = 'quantity_' . $year; $co2e_year = (floatval($row[$qty_col] ?? 0) * $ef) / 1000; $categories_summary['s2_other_energy'][$year] += $co2e_year; $categories_summary['total_scope2'][$year] += $co2e_year; } }
                        } mysqli_stmt_close($stmt_shc);
                    } else { 
						$errors[] = "Error S2 SHC details query: " . mysqli_error($conn); 
					}
                }
            } // End if ($scope2_emission_id)

            // 6. FINAL CALCULATIONS FOR SUMMARY
            foreach ($years as $year) {
                $categories_summary['total_scope1_plus_2'][$year] = ($categories_summary['total_scope1'][$year] ?? 0.0) + ($categories_summary['total_scope2'][$year] ?? 0.0);
				
				$total_s1 = $categories_summary['total_scope1'][$year] ?? 0.0;
                $total_s2 = $categories_summary['total_scope2'][$year] ?? 0.0;
                $revenue_for_this_year = $annual_revenues_by_year[$year] ?? null;

                if ($revenue_for_this_year !== null && $revenue_for_this_year > 0) {
                    $divisor_revenue = $revenue_for_this_year / 1000000.0; // For "metric tonnes CO2e/million RM revenue"
                    if ($divisor_revenue > 0) { 
                        $intensity_metrics['scope1_intensity'][$year] = $total_s1 / $divisor_revenue;
                        $intensity_metrics['scope2_intensity'][$year] = $total_s2 / $divisor_revenue;
                        $intensity_metrics['total_intensity'][$year] = ($total_s1 + $total_s2) / $divisor_revenue;
                    }
                }
            }

            // 7. PREPARE CHART DATA
            $line_chart_scope1_data_admin = json_encode(array_values($categories_summary['total_scope1']));
            $line_chart_scope2_data_admin = json_encode(array_values($categories_summary['total_scope2']));
            $line_chart_total_data_admin = json_encode(array_values($categories_summary['total_scope1_plus_2']));
            $s1_stat_pie_labels_admin = json_encode(array_values(array_keys($s1_stationary_breakdown_admin)));
            $s1_stat_pie_data_admin = json_encode(array_values($s1_stationary_breakdown_admin));
            $s1_mob_pie_labels_admin = json_encode(array_values(array_keys($s1_mobile_breakdown_admin)));
            $s1_mob_pie_data_admin = json_encode(array_values($s1_mobile_breakdown_admin));
            $s1_fug_pie_labels_admin = json_encode(array_values(array_keys($s1_fugitive_breakdown_admin)));
            $s1_fug_pie_data_admin = json_encode(array_values($s1_fugitive_breakdown_admin));
        } else {
            if (empty($errors)) { // Only show this if no DB error already occurred
                $errors[] = "Submission general details not found for ID: " . htmlspecialchars($view_id);
            }
        }
    } else { 
		$errors[] = "Invalid Submission ID."; 
	}
} else {
    // Fetch all general_info records for the list view
    $sql_list = "SELECT gi.id, gi.created_at, 
                   s1.total_scope1_emissions, 
                   s2.total_scope2_emissions
            FROM general_info gi
            LEFT JOIN scope1_emissions s1 ON gi.id = s1.general_id
            LEFT JOIN scope2_emissions s2 ON gi.id = s2.general_id
            ORDER BY gi.created_at DESC";
    $result_list = mysqli_query($conn, $sql_list);
    if ($result_list) {
        while ($row = mysqli_fetch_assoc($result_list)) {
            $submissions[] = $row;
        }
    } else {
        $errors[] = "Error fetching submissions list: " . mysqli_error($conn);
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title><?php echo htmlspecialchars($page_title); ?></title>
    <link rel="stylesheet" href="../assets/css/main.css">
    <link rel="stylesheet" href="assets/css/admin_style.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        #calculatedSummarySection { display: none; margin-top: 30px; padding-top: 20px; border-top: 2px dashed #ccc; }
		table th {color: #000; font-weight: bold}
        .summary-table { width: 100%; border-collapse: collapse; margin-bottom: 20px; font-size: 0.9em; }
        .summary-table th, .summary-table td { border: 1px solid #ccc; padding: 6px; text-align: left; }
        .summary-table th { background-color: #f0f0f0; font-weight: bold; }
        .summary-table td.year-data { text-align: right; }
        .chart-container-admin { width: 48%; margin-bottom: 20px; display: inline-block; vertical-align: top; border:1px solid #eee; padding:10px; box-sizing: border-box; min-height: 300px;}
        .full-width-chart-container-admin { width: 98%; margin-bottom: 20px; border:1px solid #eee; padding:10px; min-height: 350px; }
        @media (max-width: 900px) { .chart-container-admin { width: 98%; } }
        .section-title-admin { font-size: 1.2em; margin-top: 20px; margin-bottom: 10px; border-bottom: 1px solid #ccc; padding-bottom: 5px; font-weight: bold;}
        .details-section { margin-top: 20px; padding: 15px; background-color: #f9f9f9; border-radius: 5px; }
        .errors { color:red; border:1px solid red; padding:10px; margin-bottom:15px; }
    </style>
</head>
<body>
    <?php include_once 'includes/header.php'; ?>
    <div class="admin-wrapper">
        <nav class="admin-nav">
             <ul>
                <li><a href="dashboard.php">Dashboard</a></li>
                <li><a href="manage_admins.php">Manage Admins</a></li>
                <li><a href="view_submissions.php" class="active">View GHG Submissions</a></li>
                <li><a href="logout.php">Logout</a></li>
            </ul>
        </nav>
        <main class="admin-content">
            <h1><?php echo htmlspecialchars($page_title); ?></h1>

            <?php if (!empty($errors)): ?>
                <div class="errors"><ul><?php foreach ($errors as $error): echo "<li>".htmlspecialchars($error)."</li>"; endforeach; ?></ul></div>
            <?php endif; ?>

            <?php if (isset($_GET['view_id']) && $submission_details_general && empty($errors)): // VIEWING DETAILS OF A SINGLE SUBMISSION ?>
                <div class="details-section">
                    <p>
                        <a href="view_submissions.php" class="button small">Back to Submissions List</a>
                        <button type="button" id="toggleSummaryBtn" class="button primary small" style="margin-left: 10px;">Show Calculated Summary & Charts</button>
                    </p>
                    <h2>Details for Submission ID: <?php echo htmlspecialchars($submission_details_general['id']); ?></h2>
                    
                    <div class="section-title-admin">General Information</div>
                     <table>
						<tbody>
                        <?php foreach ($submission_details_general as $key => $value): ?>
							<tr>
								<th style="text-transform: capitalize;"><?php echo htmlspecialchars(str_replace('_', ' ', $key)); ?></th>
								<td><?php echo htmlspecialchars($value ?? 'N/A'); ?></td>
							</tr>
                        <?php endforeach; ?>
						</tbody>
					</table>

                    <div class="section-title-admin">Raw Data Entries</div>
                    <?php if ($scope1_main_data_from_db): // Check if scope 1 main data exists ?>
                        <h3>Scope 1 Emissions (Raw Entries)</h3>
                        <p><em>(Stored Total Scope 1 CO2e: <?php echo htmlspecialchars(number_format($scope1_main_data_from_db['total_scope1_emissions'] ?? 0, 4)); ?> tonnes)</em></p>

                        <h4>Stationary Combustion Entries</h4>
                        <?php if (!empty($raw_scope1_stationary_details)): ?>
                            <table class="summary-table">
								<thead>
									<tr>
										<th>Fuel</th>
										<th>Unit</th>
										<th>2021</th>
										<th>2022</th>
										<th>2023</th>
										<th>2024</th>
									</tr>
								</thead>
							<tbody>
								<?php foreach ($raw_scope1_stationary_details as $item): ?>
								<tr>
									<td><?php echo htmlspecialchars($item['fuel_type']);?></td>
									<td><?php echo htmlspecialchars($item['input_unit']);?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2021'],2));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2022'],2));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2023'],2));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2024'],2));?></td>
								</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
                        <?php else: echo "<p>No raw stationary combustion data.</p>"; endif; ?>
                        
                        <h4>Mobile Combustion Entries</h4>
                        <?php if (!empty($raw_scope1_mobile_details)): ?>
                        <table class="summary-table">
							<thead>
								<tr>
									<th>Fuel</th>
									<th>Unit</th>
									<th>2021</th>
									<th>2022</th>
									<th>2023</th>
									<th>2024</th>
								</tr>
							</thead>
							<tbody>
							<?php foreach ($raw_scope1_mobile_details as $item): ?>
								<tr>
									<td><?php echo htmlspecialchars($item['fuel_type']);?></td>
									<td><?php echo htmlspecialchars($item['input_unit']);?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2021'],2));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2022'],2));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2023'],2));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2024'],2));?></td>
								</tr>
							<?php endforeach; ?>
							</tbody>
						</table>
                        <?php else: echo "<p>No raw mobile combustion data.</p>"; endif; ?>

                        <h4>Fugitive Emissions Entries</h4>
                        <?php if (!empty($raw_scope1_fugitive_details)): ?>
                        <table class="summary-table">
							<thead>
								<tr>
									<th>Gas</th>
									<th>Unit</th>
									<th>2021</th>
									<th>2022</th>
									<th>2023</th>
									<th>2024</th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ($raw_scope1_fugitive_details as $item): ?>
								<tr>
									<td><?php echo htmlspecialchars($item['gas_type']);?></td>
									<td><?php echo htmlspecialchars($item['input_unit']);?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2021'],4));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2022'],4));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2023'],4));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2024'],4));?></td>
								</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
                        <?php else: echo "<p>No raw fugitive emission data.</p>"; endif; ?>
                    <?php else: echo "<p>No Scope 1 data was submitted or found for this record.</p>"; endif; ?>
                    
                    <hr style="margin: 20px 0;">

                    <?php if ($scope2_main_data_from_db): // Check if scope 2 main data exists ?>
                        <h3>Scope 2 Emissions (Raw Entries)</h3>
                        <p><em>(Stored Total Scope 2 CO2e: <?php echo htmlspecialchars(number_format($scope2_main_data_from_db['total_scope2_emissions'] ?? 0, 4)); ?> tonnes)</em></p>

                        <h4>Purchased Electricity Entries</h4>
                        <?php if (!empty($raw_scope2_electricity_details)): ?>
                        <table class="summary-table">
							<thead>
								<tr>
									<th>Region</th>
									<th>Unit</th>
									<th>2021</th>
									<th>2022</th>
									<th>2023</th>
									<th>2024</th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ($raw_scope2_electricity_details as $item): ?>
								<tr>
									<td><?php echo htmlspecialchars($item['region']);?></td>
									<td>kWh</td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2021'],2));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2022'],2));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2023'],2));?></td>
									<td><?php echo htmlspecialchars(number_format($item['quantity_2024'],2));?></td>
								</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
                        <?php else: echo "<p>No raw purchased electricity data.</p>"; endif; ?>

                        <?php if ($scope2_main_data_from_db['include_steam_heat_cooling']): ?>
                            <h4>Purchased Steam, Heat & Cooling Entries</h4>
                            <?php if (!empty($raw_scope2_shc_details)): ?>
                            <table class="summary-table">
								<thead>
									<tr>
										<th>Type</th>
										<th>Unit</th>
										<th>2021</th>
										<th>2022</th>
										<th>2023</th>
										<th>2024</th>
									</tr>
								</thead>
								<tbody>
									<?php foreach ($raw_scope2_shc_details as $item): ?>
									<tr>
										<td><?php echo htmlspecialchars($item['energy_type']);?></td>
										<td>metric tonnes</td>
										<td><?php echo htmlspecialchars(number_format($item['quantity_2021'],2));?></td>
										<td><?php echo htmlspecialchars(number_format($item['quantity_2022'],2));?></td>
										<td><?php echo htmlspecialchars(number_format($item['quantity_2023'],2));?></td>
										<td><?php echo htmlspecialchars(number_format($item['quantity_2024'],2));?></td>
									</tr>
									<?php endforeach; ?>
								</tbody>
							</table>
                            <?php else: echo "<p>No raw steam, heat, or cooling data (though section was included).</p>"; endif; ?>
                        <?php else: echo "<p>Steam, Heat & Cooling section was not included in this submission.</p>"; ?>
                        <?php endif; ?>
                    <?php else: echo "<p>No Scope 2 data was submitted or found for this record.</p>"; endif; ?>
					
					<hr style="margin: 20px 0;">
					
					<h3>Intensity (Raw Entries)</h3>
					<?php 
					// Check if any revenue data was actually entered/fetched for any year
					$has_revenue_data = false;
					if (isset($annual_revenues_by_year) && is_array($annual_revenues_by_year)) {
						foreach ($annual_revenues_by_year as $revenue_value) {
							if ($revenue_value !== null) {
								$has_revenue_data = true;
								break;
							}
						}
					}
					?>
					<?php if ($has_revenue_data): ?>
						<table class="summary-table">
							<thead>
								<tr>
									<th>Year</th>
									<th>Annual Revenue (RM)</th>
								</tr>
							</thead>
							<tbody>
								<?php foreach ($years as $year): ?>
								<tr>
									<td><?php echo $year; ?></td>
									<td class='year-data' style="text-align:left">
										<?php 
										$revenue_value_for_year = $annual_revenues_by_year[$year] ?? null;
										echo ($revenue_value_for_year !== null) ? htmlspecialchars(number_format($revenue_value_for_year, 2)) : '-'; 
										?>
									</td>
								</tr>
								<?php endforeach; ?>
							</tbody>
						</table>
					<?php else: ?>
						<p>No annual revenue data was submitted for this record.</p>
					<?php endif; ?>

                    <div id="calculatedSummarySection">
                        <hr style="margin-top: 40px; margin-bottom: 30px; border-style:dashed;">
                        <div class="section-title-admin">Calculated Emissions Summary & Charts</div>
                        <div class="section-title-admin" style="font-size: 1.1em; margin-top:10px;">Total GHG Emissions</div>
                        <table class="summary-table">
                            <thead>
								<tr>
									<th>Category</th>
									<th>Unit</th><?php foreach ($years as $year) echo "<th>{$year}</th>"; ?>
								</tr>
							</thead>
                            <tbody>
                                <tr>
									<td>Total Scope 1</td>
									<td>metric tonnes of CO2e</td><?php foreach ($years as $year) echo "<td class='year-data'>" . number_format($categories_summary['total_scope1'][$year], 4) . "</td>"; ?>
								</tr>
                                <tr>
									<td>Total Scope 2</td>
									<td>metric tonnes of CO2e</td><?php foreach ($years as $year) echo "<td class='year-data'>" . number_format($categories_summary['total_scope2'][$year], 4) . "</td>"; ?>
								</tr>
                                <tr style="font-weight: bold;">
									<td>Total Scope 1+2</td>
									<td>metric tonnes of CO2e</td><?php foreach ($years as $year) echo "<td class='year-data'>" . number_format($categories_summary['total_scope1_plus_2'][$year], 4) . "</td>"; ?>
								</tr>
                            </tbody>
                        </table>
						
						<div class="section-title-admin" style="font-size: 1.1em;">Intensity Metrics</div>
                        <table class="summary-table">
                            <thead>
                                <tr>
                                    <th>Category</th>
                                    <th>Unit</th>
                                    <?php foreach ($years as $y) echo "<th>{$y}</th>"; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php $intensityUnitDisplay = 'metric tonnes CO2e/million RM revenue'; ?>
                                <tr>
                                    <td>Scope 1 Intensity</td>
                                    <td><?php echo $intensityUnitDisplay; ?></td>
                                    <?php foreach ($years as $y): ?>
                                        <td class='year-data'>
                                            <?php echo ($intensity_metrics['scope1_intensity'][$y] !== null) ? htmlspecialchars(number_format($intensity_metrics['scope1_intensity'][$y], 2)) : '-'; ?>
                                        </td>
                                    <?php endforeach; ?>
                                </tr>
                                <tr>
                                    <td>Scope 2 Intensity</td>
                                    <td><?php echo $intensityUnitDisplay; ?></td>
                                    <?php foreach ($years as $y): ?>
                                        <td class='year-data'>
                                            <?php echo ($intensity_metrics['scope2_intensity'][$y] !== null) ? htmlspecialchars(number_format($intensity_metrics['scope2_intensity'][$y], 2)) : '-'; ?>
                                        </td>
                                    <?php endforeach; ?>
                                </tr>
                                <tr style="font-weight: bold;">
                                    <td>Total Intensity (Scope 1+2)</td>
                                    <td><?php echo $intensityUnitDisplay; ?></td>
                                    <?php foreach ($years as $y): ?>
                                        <td class='year-data'>
                                            <?php echo ($intensity_metrics['total_intensity'][$y] !== null) ? htmlspecialchars(number_format($intensity_metrics['total_intensity'][$y], 2)) : '-'; ?>
                                        </td>
                                    <?php endforeach; ?>
                                </tr>
                            </tbody>
                        </table>
						
                        <div class="section-title-admin" style="font-size: 1.1em;">Scope 1 Breakdown</div>
                        <table class="summary-table">
                             <thead>
								<tr>
									<th>Category</th>
									<th>Unit</th><?php foreach ($years as $year) echo "<th>{$year}</th>"; ?>
								</tr>
							</thead>
                            <tbody>
                                <tr>
									<td>Stationary Combustion</td>
									<td>metric tonnes of CO2e</td><?php foreach ($years as $year) echo "<td class='year-data'>" . number_format($categories_summary['s1_stationary'][$year], 4) . "</td>"; ?>
								</tr>
                                <tr>
									<td>Mobile Combustion</td><td>metric tonnes of CO2e</td><?php foreach ($years as $year) echo "<td class='year-data'>" . number_format($categories_summary['s1_mobile'][$year], 4) . "</td>"; ?>
								</tr>
                                <tr>
									<td>Fugitive Emissions (Refrigerants)</td><td>metric tonnes of CO2e</td><?php foreach ($years as $year) echo "<td class='year-data'>" . number_format($categories_summary['s1_fugitive'][$year], 4) . "</td>"; ?>
								</tr>
                            </tbody>
                        </table>
                        <div class="section-title-admin" style="font-size: 1.1em;">Scope 2 Breakdown</div>
                        <table class="summary-table">
                            <thead>
								<tr>
									<th>Category</th>
									<th>Unit</th><?php foreach ($years as $year) echo "<th>{$year}</th>"; ?>
								</tr>
							</thead>
                            <tbody>
                                <tr>
									<td>Electricity Consumption</td>
									<td>metric tonnes of CO2e</td><?php foreach ($years as $year) echo "<td class='year-data'>" . number_format($categories_summary['s2_electricity'][$year], 4) . "</td>"; ?>
								</tr>
                            </tbody>
                        </table>
                        <div class="section-title-admin">Summary Charts</div>
                        <div class="full-width-chart-container-admin"><canvas id="adminGhgByScopeChart"></canvas></div>
                        <div class="chart-container-admin"><canvas id="adminStationaryBreakdownChart"></canvas></div>
                        <div class="chart-container-admin"><canvas id="adminMobileBreakdownChart"></canvas></div>
                    </div>
                </div> 
            <?php else: // Show the list of submissions (this part is when $_GET['view_id'] is NOT set) ?>
                 <h3>GHG Submissions List</h3>
                <table>
                    <thead>
						<tr>
							<th>Submission ID</th>
							<th>Submitted At</th>
							<th>Total Scope 1 (tCO2e)</th>
							<th>Total Scope 2 (tCO2e)</th>
							<th>Actions</th>
						</tr>
					</thead>
                    <tbody>
                        <?php if (empty($submissions)): ?>
                            <tr><td colspan="5">No submissions found.</td></tr>
                        <?php else: ?>
                            <?php foreach ($submissions as $sub): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($sub['id']); ?></td>
                                <td><?php echo htmlspecialchars($sub['created_at']); ?></td>
                                <td><?php echo htmlspecialchars(number_format($sub['total_scope1_emissions'] ?? 0, 4)); ?></td>
                                <td><?php echo htmlspecialchars(number_format($sub['total_scope2_emissions'] ?? 0, 4)); ?></td>
                                <td><a href="view_submissions.php?view_id=<?php echo $sub['id']; ?>" class="button small">View Details & Summary</a></td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            <?php endif; // End of main if/else for view_id ?>
        </main>
    </div>
    <?php include_once 'includes/footer.php'; ?>
<script>
document.addEventListener('DOMContentLoaded', function () {
    const summarySection = document.getElementById('calculatedSummarySection');
    const toggleBtn = document.getElementById('toggleSummaryBtn');
    let chartsInitialized = false;

    if (toggleBtn && summarySection) {
        toggleBtn.addEventListener('click', function() {
            if (summarySection.style.display === 'none' || summarySection.style.display === '') {
                summarySection.style.display = 'block';
                toggleBtn.textContent = 'Hide Calculated Summary & Charts';
                if (!chartsInitialized && <?php echo (isset($_GET['view_id']) && $submission_details_general && empty($errors)) ? 'true' : 'false'; ?>) {
                    initializeAdminCharts();
                    chartsInitialized = true;
                }
            } else {
                summarySection.style.display = 'none';
                toggleBtn.textContent = 'Show Calculated Summary & Charts';
            }
        });
    }

    function initializeAdminCharts() {
        <?php if (isset($_GET['view_id']) && $submission_details_general && empty($errors)): ?>
        
        const ctxGhgByScopeAdmin = document.getElementById('adminGhgByScopeChart')?.getContext('2d');
        if (ctxGhgByScopeAdmin) {
            new Chart(ctxGhgByScopeAdmin, {
                type: 'line', data: { labels: <?php echo $line_chart_labels_admin; ?>,
                    datasets: [
                        { label: 'Total Scope 1 (tCO2e)', data: <?php echo $line_chart_scope1_data_admin; ?>, borderColor: 'rgb(54, 162, 235)', tension: 0.1, fill: false },
                        { label: 'Total Scope 2 (tCO2e)', data: <?php echo $line_chart_scope2_data_admin; ?>, borderColor: 'rgb(255, 159, 64)', tension: 0.1, fill: false },
                        { label: 'Total Scope 1+2 (tCO2e)', data: <?php echo $line_chart_total_data_admin; ?>, borderColor: 'rgb(75, 192, 192)', tension: 0.1, fill: false }
                    ]
                }, options: { responsive: true, plugins: { title: { display: true, text: 'GHG Emissions by Scope (Admin View)' } }, scales: { y: { beginAtZero: true, title: { display: true, text: 'tCO2e' } } } }
            });
        }

        function getPieColorsAdmin(count) { const baseColors = ['rgba(255, 99, 132, 0.8)','rgba(54, 162, 235, 0.8)','rgba(255, 206, 86, 0.8)','rgba(75, 192, 192, 0.8)','rgba(153, 102, 255, 0.8)','rgba(255, 159, 64, 0.8)']; let colors = []; for(let i = 0; i < count; i++) colors.push(baseColors[i % baseColors.length]); return colors; }

        const s1StatLabelsAdmin = <?php echo $s1_stat_pie_labels_admin; ?>; const s1StatDataAdmin = <?php echo $s1_stat_pie_data_admin; ?>;
        const ctxStationaryAdmin = document.getElementById('adminStationaryBreakdownChart')?.getContext('2d');
        if (ctxStationaryAdmin && s1StatLabelsAdmin.length > 0 && s1StatDataAdmin.some(d => d > 0)) {
            new Chart(ctxStationaryAdmin, { type: 'pie', data: { labels: s1StatLabelsAdmin, datasets: [{ data: s1StatDataAdmin, backgroundColor: getPieColorsAdmin(s1StatLabelsAdmin.length) }] }, options: { responsive: true, maintainAspectRatio: false, plugins: { title: { display: true, text: '<?php echo $breakdown_year_admin; ?> Stationary Breakdown (tCO2e)' } } } });
        } else if(ctxStationaryAdmin) { ctxStationaryAdmin.canvas.parentElement.innerHTML = '<p style="text-align:center; padding-top:50px;">No stationary data for chart.</p>'; }

        const s1MobLabelsAdmin = <?php echo $s1_mob_pie_labels_admin; ?>; const s1MobDataAdmin = <?php echo $s1_mob_pie_data_admin; ?>;
        const ctxMobileAdmin = document.getElementById('adminMobileBreakdownChart')?.getContext('2d');
        if (ctxMobileAdmin && s1MobLabelsAdmin.length > 0 && s1MobDataAdmin.some(d => d > 0)) {
             new Chart(ctxMobileAdmin, { type: 'pie', data: { labels: s1MobLabelsAdmin, datasets: [{ data: s1MobDataAdmin, backgroundColor: getPieColorsAdmin(s1MobLabelsAdmin.length) }] }, options: { responsive: true, maintainAspectRatio: false, plugins: { title: { display: true, text: '<?php echo $breakdown_year_admin; ?> Mobile Breakdown (tCO2e)' } } } });
        } else if(ctxMobileAdmin) { ctxMobileAdmin.canvas.parentElement.innerHTML = '<p style="text-align:center; padding-top:50px;">No mobile data for chart.</p>'; }

        const s1FugLabelsAdmin = <?php echo $s1_fug_pie_labels_admin; ?>; const s1FugDataAdmin = <?php echo $s1_fug_pie_data_admin; ?>;
        const ctxFugitiveAdmin = document.getElementById('adminFugitiveBreakdownChart')?.getContext('2d');
        if (ctxFugitiveAdmin && s1FugLabelsAdmin.length > 0 && s1FugDataAdmin.some(d => d > 0)) {
             new Chart(ctxFugitiveAdmin, { type: 'pie', data: { labels: s1FugLabelsAdmin, datasets: [{ data: s1FugDataAdmin, backgroundColor: getPieColorsAdmin(s1FugLabelsAdmin.length) }] }, options: { responsive: true, maintainAspectRatio: false, plugins: { title: { display: true, text: '<?php echo $breakdown_year_admin; ?> Fugitive Breakdown (tCO2e)' } } } });
        } else if(ctxFugitiveAdmin) { ctxFugitiveAdmin.canvas.parentElement.innerHTML = '<p style="text-align:center; padding-top:50px;">No fugitive data for chart.</p>'; }
        <?php endif; ?>
    } 
});
</script>
</body>
</html>